<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Application;
use Application\Core\Model\ApplicationSite;
use Application\Core\Model\Manager\Exception\ApplicationManagerNotFoundException;

/**
 * Class ApplicationManager
 */
class ApplicationManager implements ApplicationManagerInterface
{
    /**
     * @var SiteInterface|\Application\Core\Model\Site
     */
    protected $site;

    /**
     * @var \Application\Core\Model\Application[]
     */
    protected $enabled_apps;

    /**
     * @var \Application\Core\Model\Application[]
     */
    protected $installed_apps;

    /**
     * @param SiteInterface $site
     */
    public function __construct(SiteInterface $site)
    {
        $this->site = $site;
    }

    /**
     * {@inheritdoc}
     */
    public function allEnabled()
    {
        if ($this->enabled_apps === null) {
            $options = array(
                'select' => '`from`.*',
                'joins'  => 'INNER JOIN ' . ApplicationSite::tableName() . ' AS appsite ON `from`.id = appsite.app_id',
                'conditions' => array(
                    'appsite.site_id = :site',
                    'site'      => $this->site->id,
                ),
                'order' => '`from`.load_order',
            );

            $this->enabled_apps = $this->getApps($options);
        }

        return $this->enabled_apps;
    }

    /**
     * {@inheritdoc}
     */
    public function allInstalled()
    {
        if ($this->installed_apps === null) {
            $options = array(
                'order'  => 'load_order',
            );

            $this->installed_apps = $this->getApps($options);
        }

        return $this->installed_apps;
    }

    /**
     * {@inheritdoc}
     */
    public function isInstalled($app_name)
    {
        return array_key_exists($app_name, $this->allInstalled());
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabled($app_name)
    {
        return array_key_exists($app_name, $this->allEnabled());
    }

    /**
     * {@inheritdoc}
     */
    public function createApplication($app_name)
    {
        if ($this->findByAppName($app_name) === null) {
            Application::create(
                array(
                    'app_name'   => $app_name,
                    'load_order' => Application::max('load_order') + 10,
                )
            );
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplication($app_name)
    {
        if ($app = $this->findByAppName($app_name)) {
            if ($app->total_installations === 0) {
                $app->delete();
            }
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function createApplicationSite($app_name)
    {
        if ($app = $this->findByAppName($app_name)) {
            ApplicationSite::create(
                array(
                    'app_id'  => $app->id,
                    'site_id' => $this->site->id,
                )
            );
        } else {
            throw new ApplicationManagerNotFoundException();
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplicationSite($app_name)
    {
        $options = array(
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . Application::tableName() . ' AS a ON a.id = `from`.app_id',
            'conditions' => array(
                'a.app_name = :name AND `from`.site_id = :site',
                'name' => $app_name,
                'site' => $this->site->id,
            )
        );

        if ($app_site = ApplicationSite::first($options)) {
            $app_site->delete();
        }

        $this->clearCache();
    }

    /**
     * {@inheritdoc}
     */
    public function findByAppName($app_name)
    {
        $apps = $this->allInstalled();

        if (array_key_exists($app_name, $apps)) {
            return $apps[$app_name];
        }

        return null;
    }

    /**
     * @param array $options
     * @return \Application\Core\Model\Application[]
     */
    protected function getApps(array $options = array())
    {
        $apps = array();

        /** @var \Application\Core\Model\Application $app */
        foreach (Application::all($options) as $app) {
            $apps[$app->getName()] = $app;
        }

        return $apps;
    }

    /**
     * Resetta la cache dei modelli
     */
    protected function clearCache()
    {
        $this->enabled_apps = null;
        $this->installed_apps = null;
    }
}
