<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\SiteInterface;
use Pongho\Core\Kernel;

/**
 * Modello per le lingue.
 *
 * @property int    $id
 * @property string $iso
 * @property string $culture
 * @property string $name
 * @property string $locale
 * @property string $timezone
 */
class Language extends Base implements LanguageInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'languages';

    /**
     * Restituisce tutte le lingue abilitate del sito indicato.
     *
     * @param int $site_id
     * @return Language[]
     * @deprecated
     */
    public static function findAllBySite($site_id)
    {
        return static::all(
            array(
                'select'     => 'ls.*, ls.id AS language_site_id, `from`.*',
                'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.language_id = `from`.id',
                'conditions' => array(
                    'ls.site_id = :site AND ls.is_enabled = :enabled',
                    'site'    => $site_id,
                    'enabled' => true
                ),
            )
        );
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getIso()
    {
        return $this->iso;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getCulture()
    {
        return $this->culture;
    }

    /**
     * @return string
     */
    public function getLocale()
    {
        return $this->locale;
    }

    /**
     * @return string
     */
    public function getTimezone()
    {
        return $this->timezone;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageSite($site)
    {
        if (is_numeric($site)) {
            $site_id = intval($site);
        } elseif ($site instanceof SiteInterface) {
            $site_id = $site->getId();
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Site must be an integer or an instance of SiteInterface. "%s" given instead.',
                    var_to_string($site)
                )
            );
        }

        return Kernel::instance()->getContainer()->getService('language_site_manager')->findByLanguageAndSite($this->id, $site_id);
    }
}
