<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\ImageInterface;
use Application\Core\Model\Manager\SiteManagerInterface;
use Pongho\Core\Kernel;
use Pongho\Utilities\Inflector;

/**
 * Modello per i file.
 *
 * @property \Application\Core\Model\Site $site
 * @property int                          $id
 * @property int                          $site_id
 * @property string                       $name
 * @property int                          $size
 * @property string                       $mimetype
 * @property string                       $file
 * @property string                       $path
 */
class File extends Base implements ImageInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'files';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     * @static
     */
    public static $belongs_to = array(
        array('site', 'model' => 'Application\\Core\\Model\\Site'),
    );

    /**
     * Callback `before_save`.
     *
     * @var array
     * @static
     */
    public static $before_save = array('fixPath');

    /**
     * {@inheritdoc}
     *
     * @param boolean $delete_file Indica se eliminare o meno il file associato.
     */
    public function delete($delete_file = true)
    {
        if ($delete_file) {
            $file_path = $this->filepath();

            if (file_exists($file_path)) {
                unlink($file_path);
            }

            $view = Kernel::instance()->getContainer()->getService('theme_view');

            /** @var SiteManagerInterface $site_manager */
            $site_manager = Kernel::instance()->getContainer()->getService('site_manager');

            foreach ($view->getOption('image_sizes', array()) as $size => $opts) {
                $path = $site_manager->findById($this->site_id)->getImagesPath($this->addSize($size));

                if (file_exists($path)) {
                    unlink($path);
                }
            }
        }

        return parent::delete();
    }

    /**
     * Cerca un file in base al nome ed al percorso (campi `file` e `path` in tabella).
     *
     * @param string  $name    Nome del file caricato.
     * @param string  $path    Cartella di destinazione del file.
     * @param integer $site_id ID del sito di riferimento.
     * @return File
     */
    public static function findByName($name, $path, $site_id)
    {
        $path = '/' . trim($path, '/');

        return static::first(
            array(
                'conditions' => array(
                    'file = :file AND path = :path AND site_id = :site',
                    'file' => $name,
                    'path' => $path,
                    'site' => $site_id
                )
            )
        );
    }

    /**
     * {@inheritdoc}
     */
    public function path()
    {
        /** @var SiteManagerInterface $site_manager */
        $site_manager = Kernel::instance()->getContainer()->getService('site_manager');

        return $site_manager->findById($this->site_id)->getUploadsUrl($this->getFixedPath() . '/' . $this->file);
    }

    /**
     * {@inheritdoc}
     */
    public function filepath()
    {
        /** @var SiteManagerInterface $site_manager */
        $site_manager = Kernel::instance()->getContainer()->getService('site_manager');

        return $site_manager->findById($this->site_id)->getUploadsPath($this->getFixedPath() . '/' . $this->file);
    }

    /**
     * {@inheritdoc}
     */
    public function size()
    {
        return Inflector::sizize($this->size);
    }

    /**
     * Corregge il percorso del file corrente.
     *
     * Si assicura che il percorso `uploads_path` del sito non sia presente.
     *
     * Questo metodo è lanciato in automatico al momento del salvataggio del modello.
     */
    public function fixPath()
    {
        $this->path = $this->getFixedPath();
    }

    /**
     * Restituisce il percorso del file corrente, senza il percorso `uploads_path` del sito.
     *
     * Questo metodo non modifica il valore dell’attributo `path` del modello.
     *
     * @return string
     */
    public function getFixedPath()
    {
        /** @var SiteManagerInterface $site_manager */
        $site_manager = Kernel::instance()->getContainer()->getService('site_manager');

        $uploads_path = $site_manager->findById($this->site_id)->getUploadsPath();

        if (strpos($this->path, $uploads_path) === 0) {
            return substr($this->path, strlen($uploads_path));
        }

        return $this->path;
    }

    /**
     * {@inheritdoc}
     */
    public function addSize($size)
    {
        $info = pathinfo($this->file);

        return $this->path . '/' . $info['filename'] . '-' . $size . '.' . $info['extension'];
    }
}
