<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\ApplicationInterface;

/**
 * Application.
 *
 * @property integer                                   $id
 * @property string                                    $app_name
 * @property integer                                   $load_order
 * @property integer                                   $total_installations
 * @property \Application\Core\Model\ApplicationSite[] $installations
 */
class Application extends Base implements ApplicationInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'apps';

    /**
     * @var array
     */
    public static $has_many = array(
        array('installations', 'model' => 'Application\\Core\\Model\\ApplicationSite', 'foreign_key' => 'app_id'),
    );

    /**
     * @var array
     */
    public static $count_has_many = array(
        array('total_installations', 'model' => 'Application\\Core\\Model\\ApplicationSite', 'foreign_key' => 'app_id'),
    );

    /**
     * findAllEnabled
     *
     * @param $site_id
     * @return \Application\Core\Model\Application[]
     * @static
     */
    public static function findAllEnabled($site_id)
    {
        return self::all(
            array(
                'select' => '`from`.*',
                'joins'  => 'INNER JOIN ' . ApplicationSite::tableName() . ' AS appsite ON `from`.id = appsite.app_id',
                'conditions' => array(
                    'appsite.site_id = :site',
                    'site'      => $site_id,
                ),
                'order' => 'app_name',
            )
        );
    }

    /**
     * findByName
     *
     * @param $name
     * @return \Application\Core\Model\Application
     */
    public static function findByName($name)
    {
        return self::first(
            array(
                'conditions' => array('app_name = ?', $name),
            )
        );
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        foreach ($this->installations as $app_site) {
            if (!$app_site->delete()) {
                return false;
            }
        }

        return parent::delete();
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->app_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getVersion()
    {
        // Creo il nome del kernel
        /** @var \Pongho\Core\Plugin $kernel_class */
        $kernel_class = "Application\\{$this->app_name}\\Kernel";

        if (!class_exists($kernel_class)) {
            return 'Kernel not exists';
        }

        return $kernel_class::VERSION;
    }
}
