<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Installer;

include_once(__DIR__ . '/../../../src/functions.php');

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Schema\Comparator;
use Pongho\Utilities\DateTime;

/**
 * Class Installer
 *
 * Installa e prepara il database per Pongho
 */
class KernelInstaller extends Installer
{
    /**
     * @var \Doctrine\DBAL\Schema\Schema
     */
    protected $schema;

    /**
     * @var \Doctrine\DBAL\Schema\Schema
     */
    protected $old_schema;

    /**
     * @var \Doctrine\DBAL\Connection
     */
    protected $connection;

    /**
     * @var array
     */
    protected $languages = array();

    /**
     * @var array
     */
    protected $sites = array();

    /**
     * @var array
     */
    protected $modules = array();

    /**
     * @var array
     */
    protected $apps = array();

    /**
     * @var array
     */
    protected $roles = array();

    /**
     * @var array
     */
    protected $permits = array();

    /**
     * @var array
     */
    protected $countries = array();

    /**
     * @var array
     */
    protected $regions = array();

    /**
     * @var array
     */
    protected $provinces = array();

    /**
     * @var array
     */
    protected $users = array();

    /**
     * @var string
     */
    protected $domain;

    /**
     * @param Connection $connection
     * @param string     $domain
     */
    public function __construct(Connection $connection, $domain)
    {
        $this->connection = $connection;
        $this->domain = $domain;
    }

    /**
     * @return \Doctrine\DBAL\Schema\Schema
     */
    protected function getSchema()
    {
        /** @var \Doctrine\DBAL\Schema\Schema $schema */
        $this->schema = $this->connection->getSchemaManager()->createSchema();
        $this->old_schema = clone $this->schema;

        return $this->schema;
    }

    /**
     * Aggiorna lo schema
     */
    protected function updateSchema()
    {
        $comparator = new Comparator();
        $diff = $comparator->compare($this->old_schema, $this->schema);

        $platform = $this->connection->getDatabasePlatform();

        foreach ($diff->toSql($platform) as $query) {
            $this->connection->exec($query);
        }
    }

    /**
     * Installa Pongho
     */
    public function install()
    {
        if ($this->getSchema()->hasTable('pongho_settings')) {
            throw new \Exception('Pongho is already installed!');
        }

        $this->installSchema();

        $this->installLanguages();
        $this->installSites();
        $this->installModules();
        $this->installApps();
        $this->installCountries();
        $this->installRegions();
        $this->installProvinces();
        $this->installCities();
        $this->installRoles();
        $this->installPermits();
        $this->installUsers();

        $this->prepareBaseStuff();
        $this->installTranslations();
    }

    /**
     * Crea la struttura del database
     */
    protected function installSchema()
    {
        $schema = $this->getSchema();

        // Impostazioni
        $settings_table = $schema->createTable('pongho_settings');
        $settings_table->addColumn('key', 'string', array('length' => 50));
        $settings_table->addColumn('value', 'text', array('length' => 65535, 'default' => ''));
        $settings_table->setPrimaryKey(array('key'));

        // Lingue
        $languages_table = $schema->createTable('pongho_languages');
        $languages_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $languages_table->addColumn('iso', 'string', array('length' => 50, 'default' => ''));
        $languages_table->addColumn('culture', 'string', array('length' => 10, 'default' => ''));
        $languages_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $languages_table->addColumn('locale', 'string', array('length' => 50, 'default' => ''));
        $languages_table->addColumn('timezone', 'string', array('length' => 50, 'default' => ''));
        $languages_table->setPrimaryKey(array('id'));

        // Siti
        $sites_table = $schema->createTable('pongho_sites');
        $sites_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $sites_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $sites_table->addColumn('domain', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('content_path', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('images_path', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('images_url', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('uploads_path', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('uploads_url', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('themes_path', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('themes_url', 'string', array('length' => 200, 'default' => ''));
        $sites_table->addColumn('default_language_id', 'integer', array('unsigned' => true));
        $sites_table->addColumn('settings', 'text', array('length' => 65535, 'default' => ''));
        $sites_table->setPrimaryKey(array('id'));

        // Moduli
        $modules_table = $schema->createTable('pongho_modules');
        $modules_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $modules_table->addColumn('controller', 'string', array('length' => 100, 'default' => ''));
        // la colonna "node_type" è gestita dal Cms, controllare che non venga usata per sbaglio da altre applicazioni
        // che non dipendono dal Cms
        $modules_table->setPrimaryKey(array('id'));

        // Tabella di relazione Siti Moduli
        $modules_sites_table = $schema->createTable('pongho_modules_sites');
        $modules_sites_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $modules_sites_table->addColumn('module_id', 'integer', array('unsigned' => true));
        $modules_sites_table->addColumn('site_id', 'integer', array('unsigned' => true));
        $modules_sites_table->addColumn('settings', 'text', array('length' => 65535, 'default' => ''));
        $modules_sites_table->addColumn('is_enabled', 'boolean', array('default' => false));
        // le colonne "is_menu_enabled", "show_in_dashboard", "node_count" e "translations" sono gestite dal Cms
        // controllare che non vengano usate per sbaglio da altre applicazioni che non dipendono dal Cms
        $modules_sites_table->setPrimaryKey(array('id'));
        $modules_sites_table->addForeignKeyConstraint($modules_table, array('module_id'), array('id'));
        $modules_sites_table->addForeignKeyConstraint($sites_table, array('site_id'), array('id'));

        // Tabella di relazione Siti Lingue
        $languages_sites_table = $schema->createTable('pongho_languages_sites');
        $languages_sites_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $languages_sites_table->addColumn('language_id', 'integer', array('unsigned' => true));
        $languages_sites_table->addColumn('site_id', 'integer', array('unsigned' => true));
        $languages_sites_table->addColumn('path', 'string', array('length' => 70, 'default' => ''));
        $languages_sites_table->addColumn('is_enabled', 'boolean', array('default' => false));
        $languages_sites_table->setPrimaryKey(array('id'));
        $languages_sites_table->addForeignKeyConstraint($languages_table, array('language_id'), array('id'));
        $languages_sites_table->addForeignKeyConstraint($sites_table, array('site_id'), array('id'));

        // Tabella di relazione Lingue del sito Moduli
        $languages_modules_table = $schema->createTable('pongho_languages_modules');
        $languages_modules_table->addColumn('language_site_id', 'integer', array('unsigned' => true));
        $languages_modules_table->addColumn('module_id', 'integer', array('unsigned' => true));
        $languages_modules_table->addColumn('path', 'string', array('length' => 100, 'default' => ''));
        $languages_modules_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $languages_modules_table->addColumn('position', 'integer', array('unsigned' => true, 'default' => 0));
        $languages_modules_table->addColumn('is_enabled', 'boolean', array('default' => false));
        $languages_modules_table->setPrimaryKey(array('language_site_id', 'module_id'));
        $languages_modules_table->addForeignKeyConstraint($languages_sites_table, array('language_site_id'), array('id'));
        $languages_modules_table->addForeignKeyConstraint($modules_table, array('module_id'), array('id'));

        // File
        $files_table = $schema->createTable('pongho_files');
        $files_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $files_table->addColumn('site_id', 'integer', array('unsigned' => true));
        $files_table->addColumn('name', 'string', array('length' => 200, 'default' => ''));
        $files_table->addColumn('size', 'integer', array('unsigned' => true, 'default' => 0));
        $files_table->addColumn('mimetype', 'string', array('length' => 100, 'default' => ''));
        $files_table->addColumn('file', 'string', array('length' => 200, 'default' => ''));
        $files_table->addColumn('path', 'string', array('length' => 50, 'default' => ''));
        $files_table->setPrimaryKey(array('id'));
        $files_table->addForeignKeyConstraint($sites_table, array('site_id'), array('id'));

        // LinkType
        $link_types_table = $schema->createTable('pongho_link_types');
        $link_types_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $link_types_table->addColumn('language_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $link_types_table->addColumn('link_type', 'string', array('length' => 255, 'default' => ''));
        $link_types_table->addColumn('model_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $link_types_table->addColumn('model_path', 'string', array('length' => 255, 'default' => ''));
        $link_types_table->addColumn('handler', 'string', array('length' => 255, 'default' => ''));
        $link_types_table->setPrimaryKey(array('id'));
        $link_types_table->addForeignKeyConstraint($languages_table, array('language_id'), array('id'));

        // Applicazioni
        $apps_table = $schema->createTable('pongho_apps');
        $apps_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $apps_table->addColumn('app_name', 'string', array('length' => 50, 'default' => ''));
        $apps_table->addColumn('load_order', 'integer', array('unsigned' => true, 'default' => 0));
        $apps_table->setPrimaryKey(array('id'));

        $apps_sites_table = $schema->createTable('pongho_apps_sites');
        $apps_sites_table->addColumn('app_id', 'integer', array('unsigned' => true));
        $apps_sites_table->addColumn('site_id', 'integer', array('unsigned' => true));
        $apps_sites_table->setPrimaryKey(array('app_id', 'site_id'));
        $apps_sites_table->addForeignKeyConstraint($apps_table, array('app_id'), array('id'));
        $apps_sites_table->addForeignKeyConstraint($sites_table, array('site_id'), array('id'));

        // Traduzioni
        $translations_table = $schema->createTable('pongho_translations');
        $translations_table->addColumn('key', 'string', array('length' => 50));
        $translations_table->addColumn('language_site_id', 'integer', array('unsigned' => true));
        $translations_table->addColumn('application', 'string', array('length' => 50));
        $translations_table->addColumn('value', 'text', array('length' => 65535, 'default' => ''));
        $translations_table->setPrimaryKey(array('key', 'language_site_id'));
        $translations_table->addForeignKeyConstraint($languages_sites_table, array('language_site_id'), array('id'));

        // Ruoli
        $roles_table = $schema->createTable('pongho_roles');
        $roles_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $roles_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $roles_table->addColumn('description', 'string', array('length' => 200, 'default' => ''));
        $roles_table->setPrimaryKey(array('id'));

        // Permessi
        $permits_table = $schema->createTable('pongho_permits');
        $permits_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $permits_table->addColumn('key', 'string', array('length' => 50, 'default' => ''));
        $permits_table->addColumn('is_enabled', 'boolean', array('default' => false));
        $permits_table->setPrimaryKey(array('id'));
        $permits_table->addUniqueIndex(array('key'));

        // Relazioni ruoli - permessi
        $permits_roles_table = $schema->createTable('pongho_permits_roles');
        $permits_roles_table->addColumn('permit_id', 'integer', array('unsigned' => true));
        $permits_roles_table->addColumn('role_id', 'integer', array('unsigned' => true));
        $permits_roles_table->setPrimaryKey(array('permit_id', 'role_id'));
        $permits_roles_table->addForeignKeyConstraint($permits_table, array('permit_id'), array('id'));
        $permits_roles_table->addForeignKeyConstraint($roles_table, array('role_id'), array('id'));

        // Indirizzi - Paesi
        $countries_table = $schema->createTable('pongho_countries');
        $countries_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $countries_table->addColumn('name', 'string', array('length' => 255, 'default' => ''));
        $countries_table->addColumn('postal_name', 'string', array('length' => 20, 'default' => ''));
        $countries_table->addColumn('num', 'string', array('length' => 5, 'default' => ''));
        $countries_table->addColumn('code', 'string', array('length' => 2, 'default' => ''));
        $countries_table->addColumn('code3', 'string', array('length' => 3, 'default' => ''));
        $countries_table->addColumn('address_template', 'text', array('length' => 65535, 'default' => ''));
        $countries_table->setPrimaryKey(array('id'));

        // Indirizzi - Regioni
        $regions_table = $schema->createTable('pongho_regions');
        $regions_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $regions_table->addColumn('country_id', 'integer', array('unsigned' => true));
        $regions_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $regions_table->addColumn('name_long', 'string', array('length' => 50, 'default' => ''));
        $regions_table->addColumn('url', 'string', array('length' => 50, 'default' => ''));
        $regions_table->setPrimaryKey(array('id'));
        $regions_table->addForeignKeyConstraint($countries_table, array('country_id'), array('id'));

        // Indirizzi - Province
        $provinces_table = $schema->createTable('pongho_provinces');
        $provinces_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $provinces_table->addColumn('region_id', 'integer', array('unsigned' => true));
        $provinces_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $provinces_table->addColumn('plate', 'string', array('length' => 10, 'default' => ''));
        $provinces_table->setPrimaryKey(array('id'));
        $provinces_table->addForeignKeyConstraint($regions_table, array('region_id'), array('id'));

        // Indirizzi - Città
        $cities_table = $schema->createTable('pongho_cities');
        $cities_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $cities_table->addColumn('province_id', 'integer', array('unsigned' => true));
        $cities_table->addColumn('region_id', 'integer', array('unsigned' => true));
        $cities_table->addColumn('name', 'string', array('length' => 255, 'default' => ''));
        $cities_table->setPrimaryKey(array('id'));
        $cities_table->addForeignKeyConstraint($provinces_table, array('province_id'), array('id'));
        $cities_table->addForeignKeyConstraint($regions_table, array('region_id'), array('id'));

        // Utenti - todo: riordinare le colonne
        $users_table = $schema->createTable('pongho_users');
        $users_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $users_table->addColumn('language_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $users_table->addColumn('role_id', 'integer', array('unsigned' => true));
        $users_table->addColumn('username', 'string', array('length'  => 50, 'default' => '')); // todo: assegnare chiave univoca
        $users_table->addColumn('email', 'string', array('length'  => 50, 'default' => '')); // todo: assegnare chiave univoca
        $users_table->addColumn('password', 'string', array('length'  => 50, 'default' => '')); // todo: considerare di limitare a 32 caratteri
        $users_table->addColumn('avatar_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $users_table->addColumn('description', 'text', array('length' => 65535, 'default' => ''));
        $users_table->addColumn('activation_key', 'string', array('length' => 32, 'default' => ''));
        $users_table->addColumn('reset_password_key', 'string', array('length' => 32, 'default' => ''));
        $users_table->addColumn('fullname', 'string', array('length' => 100, 'default' => ''));
        $users_table->addColumn('is_active', 'boolean', array('default' => null, 'notnull' => false));
        $users_table->addColumn('is_founder', 'boolean', array('default' => false));
        $users_table->addColumn('is_business', 'boolean', array('default' => false));
        $users_table->addColumn('created_at', 'datetime');
        $users_table->addColumn('updated_at', 'datetime');
        $users_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('surname', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('company', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('partita_iva', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('codice_fiscale', 'string', array('length' => 16, 'default' => ''));
        $users_table->addColumn('address', 'string', array('length' => 200, 'default' => ''));
        $users_table->addColumn('city', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('postcode', 'string', array('length' => 20, 'default' => ''));
        $users_table->addColumn('province_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $users_table->addColumn('country_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $users_table->addColumn('telephone', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('fax', 'string', array('length' => 50, 'default' => ''));
        $users_table->addColumn('website', 'string', array('length' => 100, 'default' => ''));
        $users_table->addColumn('googleplus', 'string', array('length' => 255, 'default' => ''));
        $users_table->addColumn('newsletter', 'boolean', array('default' => false)); // todo: deve gestirlo la newsletter
        $users_table->addColumn('newsletter_activation_key', 'string', array('length' => 32, 'default' => ''));
        $users_table->addColumn('url', 'string', array('length' => 255, 'default' => ''));
        $users_table->addColumn('old_url', 'string', array('length' => 255, 'default' => ''));
        $users_table->addColumn('address2', 'text', array('length'  => 255, 'default' => '')); // todo: fixare, o text 65535 o string 255
        $users_table->setPrimaryKey(array('id'));
        // todo: sarebbe il caso di non appoggiarsi solo alle validazioni del modello Account
//        $users_table->addUniqueIndex(array('username'));
//        $users_table->addUniqueIndex(array('email'));
        $users_table->addIndex(array('created_at'));
        $users_table->addForeignKeyConstraint($languages_table, array('language_id'), array('id'));
        $users_table->addForeignKeyConstraint($roles_table, array('role_id'), array('id'));
        $users_table->addForeignKeyConstraint($files_table, array('avatar_id'), array('id'));
        $users_table->addForeignKeyConstraint($provinces_table, array('province_id'), array('id'));
        $users_table->addForeignKeyConstraint($countries_table, array('country_id'), array('id'));

        // Indirizzi
        $addresses_table = $schema->createTable('pongho_addresses');
        $addresses_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $addresses_table->addColumn('user_id', 'integer', array('unsigned' => true));
        $addresses_table->addColumn('name', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('surname', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('address1', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('address2', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('city', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('province_name', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('postcode', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('country_id', 'integer', array('unsigned' => true));
        $addresses_table->addColumn('telephone', 'string', array('length' => 100, 'default' => ''));
        $addresses_table->addColumn('province_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $addresses_table->setPrimaryKey(array('id'));
        $addresses_table->addForeignKeyConstraint($countries_table, array('country_id'), array('id'));
        $addresses_table->addForeignKeyConstraint($users_table, array('user_id'), array('id'));
        // todo: manca la relazione su province_id, in database DEV non c'è, non so se è per un motivo valido

        // Relazioni utenti
        $users_relations_table = $schema->createTable('pongho_users_relations');
        $users_relations_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $users_relations_table->addColumn('parent_id', 'integer', array('unsigned' => true));
        $users_relations_table->addColumn('child_id', 'integer', array('unsigned' => true));
        $users_relations_table->setPrimaryKey(array('id'));
        $users_relations_table->addUniqueIndex(array('parent_id', 'child_id'));
        $users_relations_table->addForeignKeyConstraint($users_table, array('parent_id'), array('id'));
        $users_relations_table->addForeignKeyConstraint($users_table, array('child_id'), array('id'));

        // Portafoglio utenti
        $users_portfolio_table = $schema->createTable('pongho_users_portfolio');
        $users_portfolio_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $users_portfolio_table->addColumn('relation_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $users_portfolio_table->addColumn('ancestor_id', 'integer', array('unsigned' => true));
        $users_portfolio_table->addColumn('descendant_id', 'integer', array('unsigned' => true));
        $users_portfolio_table->addColumn('depth', 'smallint', array('unsigned' => true, 'default' => 0));
        $users_portfolio_table->setPrimaryKey(array('id'));
        $users_portfolio_table->addUniqueIndex(array('relation_id', 'ancestor_id', 'descendant_id'));
        $users_portfolio_table->addForeignKeyConstraint($users_relations_table, array('relation_id'), array('id'));
        $users_portfolio_table->addForeignKeyConstraint($users_table, array('ancestor_id'), array('id'));
        $users_portfolio_table->addForeignKeyConstraint($users_table, array('descendant_id'), array('id'));

        // Notifiche - Template
        $notifications_templates_table = $schema->createTable('pongho_notifications_templates');
        $notifications_templates_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $notifications_templates_table->addColumn('site_id', 'integer', array('unsigned' => true));
        $notifications_templates_table->addColumn('notification', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('sender_to_admin', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('reply_to_admin', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('recipient_to_admin', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('sender_to_user', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('reply_to_user', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('recipient_to_user', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('subject_to_admin', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_table->addColumn('content_to_admin', 'text', array('default' => ''));
        $notifications_templates_table->setPrimaryKey(array('id'));
        $notifications_templates_table->addForeignKeyConstraint($sites_table, array('site_id'), array('id'));
        $notifications_templates_table->addUniqueIndex(array('site_id', 'notification'));

        // Notifiche - Traduzioni dei template
        $notifications_templates_translations_table = $schema->createTable('pongho_notifications_templates_translations');
        $notifications_templates_translations_table->addColumn('template_id', 'integer', array('unsigned' => true));
        $notifications_templates_translations_table->addColumn('language_id', 'integer', array('unsigned' => true));
        $notifications_templates_translations_table->addColumn('subject_to_user', 'string', array('length' => 255, 'default' => ''));
        $notifications_templates_translations_table->addColumn('content_to_user', 'text', array('default' => ''));
        $notifications_templates_translations_table->setPrimaryKey(array('template_id', 'language_id'));
        $notifications_templates_translations_table->addForeignKeyConstraint($notifications_templates_table, array('template_id'), array('id'));
        $notifications_templates_translations_table->addForeignKeyConstraint($languages_table, array('language_id'), array('id'));

        // Gruppi - todo: da spostare nella newsletter
        $groups_table = $schema->createTable('pongho_groups');
        $groups_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $groups_table->addColumn('name', 'string', array('length' => 50, 'default' => ''));
        $groups_table->addColumn('description', 'string', array('length' => 200, 'default' => ''));
        $groups_table->setPrimaryKey(array('id'));

        // Gruppi utenti - todo: da spostare nella newsletter
        $groups_users_table = $schema->createTable('pongho_groups_users');
        $groups_users_table->addColumn('group_id', 'integer', array('unsigned' => true));
        $groups_users_table->addColumn('user_id', 'integer', array('unsigned' => true));
        $groups_users_table->setPrimaryKey(array('group_id', 'user_id'));
        $groups_users_table->addForeignKeyConstraint($groups_table, array('group_id'), array('id'));
        $groups_users_table->addForeignKeyConstraint($users_table, array('user_id'), array('id'));

        // Filtri - è di Admin, ma siccome vanno in coppia (ok, non sempre) tanto vale installarla qui
        $filters_table = $schema->createTable('pongho_filters');
        $filters_table->addColumn('id', 'integer', array('unsigned' => true, 'autoincrement' => true));
        $filters_table->addColumn('name', 'string', array('length' => 255, 'default' => ''));
        $filters_table->addColumn('namespace', 'string', array('length' => 255, 'default' => ''));
        $filters_table->addColumn('filter', 'text', array('length' => 65535, 'default' => ''));
        $filters_table->addColumn('created_at', 'datetime');
        $filters_table->addColumn('updated_at', 'datetime');
        $filters_table->addColumn('user_id', 'integer', array('unsigned' => true, 'notnull' => false));
        $filters_table->setPrimaryKey(array('id'));
        $filters_table->addForeignKeyConstraint($users_table, array('user_id'), array('id'));

        $this->updateSchema();

        // Installo le funzioni di supporto nel database
        $platform = $this->connection->getDatabasePlatform()->getName();
        if ($platform === 'mysql') {
            // pongho_like()
            $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_mysql_pongho_like.sql'));
        } elseif ($platform === 'postgresql') {
            // DATE_FORMAT()
            $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_DATE_FORMAT.sql'));

            // SUBTIME()
            $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_SUBTIME.sql'));

            // fixDates()
            $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_fixDates.sql'));

            // pongho_like()
            $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_pongho_like.sql'));
        }
    }

    /**
     * Installa le lingue
     */
    protected function installLanguages()
    {
        $columns = array('iso', 'culture', 'name', 'locale', 'timezone');

        $languages = array(
            array('it', 'it_IT', 'Italiano', 'it_IT.UTF-8', 'Europe/Rome'),
            array('en', 'en_GB', 'English', 'en_GB.UTF-8', 'Europe/London'),
            array('de', 'de_DE', 'Deutsch', 'de_DE.UTF-8', 'Europe/Berlin'),
            array('fr', 'fr_FR', 'Français', 'fr_FR.UTF-8', 'Europe/Paris'),
        );

        $this->languages = array();

        foreach ($languages as $language) {
            $this->connection->insert('pongho_languages', array_combine($columns, $language));
            $this->languages[] = $this->getLastInsertId('pongho_languages');
        }
    }

    /**
     * Configura il primo sito
     */
    protected function installSites()
    {
        $data = array(
            'name'                => 'Pongho',
            'domain'              => $this->domain,
            'content_path'        => '/content',
            'images_path'         => '/content/images',
            'images_url'          => $this->domain . '/content/images',
            'uploads_path'        => '/content/uploads',
            'uploads_url'         => $this->domain . '/content/uploads',
            'themes_path'         => '/content/themes',
            'themes_url'          => $this->domain . '/content/themes',
            'default_language_id' => reset($this->languages),
            'settings'            => serialize(array()),
        );

        $this->connection->insert('pongho_sites', $data);
        $this->sites[] = $this->getLastInsertId('pongho_sites');
    }

    /**
     * Installa i moduli di base
     */
    protected function installModules()
    {
        $this->connection->insert('pongho_modules', array('controller' => 'Application\Core\Controller\UserController'));
        $this->modules['user'] = $this->getLastInsertId('pongho_modules');
    }

    /**
     * Installa le applicazioni di base (Core e Admin)
     *
     * @todo Considerare di installare Admin separatamente
     */
    protected function installApps()
    {
        $this->connection->insert(
            'pongho_apps',
            array(
                'app_name'   => 'Core',
                'load_order' => 1,
            ),
            array(
                'app_name'   => \PDO::PARAM_STR,
                'load_order' => \PDO::PARAM_INT,
            )
        );

        $this->apps['core'] = $this->getLastInsertId('pongho_apps');

        $this->connection->insert(
            'pongho_apps',
            array(
                'app_name'   => 'Admin',
                'load_order' => 2,
            ),
            array(
                'app_name'   => \PDO::PARAM_STR,
                'load_order' => \PDO::PARAM_INT,
            )
        );

        $this->apps['admin'] = $this->getLastInsertId('pongho_apps');
    }

    /**
     * Installa i ruoli
     */
    protected function installRoles()
    {
        $roles = array(
            'anonymous'  => array(
                'name'        => 'Utente non registrato',
                'description' => 'Ruolo speciale per gestire i permessi degli utenti che non hanno eseguito l’accesso al sito.'
            ),
            'registered' => array(
                'name'        => 'Utente registrato',
                'description' => 'È il ruolo che viene assegnato di default agli utenti registrati.'
            ),
            'admin'      => array(
                'name'        => 'Amministratore',
                'description' => 'Ruolo amministrativo per l’accesso all’area privata dove l’utente può gestire il sito.'
            ),
        );

        foreach ($roles as $name => $role) {
            $this->connection->insert('pongho_roles', $role);
            $this->roles[$name] = $this->getLastInsertId('pongho_roles');
        }
    }

    /**
     * Installa i permessi
     */
    protected function installPermits()
    {
        $permits = array(
            'admin'                       => true,
            'admin.options'               => true,
            'admin.options.translations'  => true,
            'admin.options.notifications' => true,
            'users.admin'                 => true,
            'users.add'                   => true,
            'users.edit'                  => true,
            'users.delete'                => true,
            'users.handle_groups'         => true, // todo: spostare in newsletter
            'users.handle_roles'          => true,
            'users.handle_assoc'          => false,
            'users.is_ancestor'           => false,
            'users.view_all_users'        => false,
        );

        $field_key_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        foreach ($permits as $key => $enabled) {
            $this->connection->insert(
                'pongho_permits',
                array(
                    $field_key_name => $key,
                    'is_enabled'    => $enabled,
                ),
                array(
                    $field_key_name => \PDO::PARAM_STR,
                    'is_enabled'    => \PDO::PARAM_BOOL,
                )
            );

            $this->permits[$key] = $this->getLastInsertId('pongho_permits');
        }
    }

    /**
     * Installa gli utenti
     */
    protected function installUsers()
    {
        $language_id = reset($this->languages);

        $date = new DateTime();
        $timestamp = $date->format('U');
        $url = (string)$timestamp . (string)random(8, '0123456789');

        $users = array(
            'anonymous' => array(
                'language_id' => $language_id,
                'role_id'     => $this->roles['anonymous'],
                'username'    => 'anonymous',
                'email'       => '',
                'password'    => '',
                'is_active'   => true,
                'is_founder'  => false,
                'fullname'    => '',
                'name'        => '',
                'surname'     => '',
                'company'     => '',
                'url'         => '',
                'created_at'  => $date->format('Y-m-d H:i:s'),
                'updated_at'  => $date->format('Y-m-d H:i:s'),
            ),
            'metaline'  => array(
                'language_id' => $language_id,
                'role_id'     => $this->roles['admin'],
                'username'    => 'metaline',
                'email'       => 'sviluppo@metaline.it',
                'password'    => '09e5049acebf849ecd8f90e3134842ca',
                'is_active'   => true,
                'is_founder'  => true,
                'fullname'    => 'Daniele De Nobili',
                'name'        => 'Daniele',
                'surname'     => 'De Nobili',
                'company'     => 'Meta Line S.r.l.',
                'url'         => $url,
                'created_at'  => $date->format('Y-m-d H:i:s'),
                'updated_at'  => $date->format('Y-m-d H:i:s'),
            ),
        );

        foreach ($users as $username => $user) {
            $this->connection->insert(
                'pongho_users', $user,
                array(
                    'language_id' => \PDO::PARAM_INT,
                    'role_id'     => \PDO::PARAM_INT,
                    'username'    => \PDO::PARAM_STR,
                    'email'       => \PDO::PARAM_STR,
                    'password'    => \PDO::PARAM_STR,
                    'is_active'   => \PDO::PARAM_BOOL,
                    'is_founder'  => \PDO::PARAM_BOOL,
                    'fullname'    => \PDO::PARAM_STR,
                    'name'        => \PDO::PARAM_STR,
                    'surname'     => \PDO::PARAM_STR,
                    'company'     => \PDO::PARAM_STR,
                    'url'         => \PDO::PARAM_STR,
                    'created_at'  => \PDO::PARAM_STR,
                    'updated_at'  => \PDO::PARAM_STR,
                )
            );
            $this->users[$username] = $this->getLastInsertId('pongho_users');
        }

    }

    /**
     * Aggiunge i Paesi, al momento solo l'Italia
     */
    protected function installCountries()
    {
        $template = <<<TEXT
{name} {surname}
{address1}
{address2}
{postcode} {city} ({province})
{country}
{telephone}
TEXT;

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/countries.json'), true) as $row) {
            // todo: Al momento gestiamo le spedizioni solo in Italia, quando avremo un migliore supporto all'I18n i dati verranno caricati tutti da file
            if ($row['name'] === 'Italia') {
                $row['postal_name'] = 'ITALY';
                $row['address_template'] = $template;
            }

            $this->connection->insert(
                'pongho_countries',
                $row,
                array(
                    'name'             => \PDO::PARAM_STR,
                    'code'             => \PDO::PARAM_STR,
                    'code3'            => \PDO::PARAM_STR,
                    'num'              => \PDO::PARAM_STR,
                    'postal_name'      => \PDO::PARAM_STR,
                    'address_template' => \PDO::PARAM_STR,
                )
            );

            $this->countries[$row['name']] = $this->getLastInsertId('pongho_countries');
        }
    }

    /**
     * Popola la tabella delle regioni.
     */
    protected function installRegions()
    {
        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/regions.json'), true) as $region) {
            // todo: Al momento ci sono solo le regioni italiane
            $region['country_id'] = $this->countries['Italia'];

            $this->connection->insert(
                'pongho_regions',
                $region,
                array(
                    'id'         => \PDO::PARAM_INT,
                    'country_id' => \PDO::PARAM_INT,
                    'name'       => \PDO::PARAM_STR,
                    'name_long'  => \PDO::PARAM_STR,
                    'url'        => \PDO::PARAM_STR,
                )
            );
        }
    }

    /**
     * Popola la tabella delle province.
     */
    protected function installProvinces()
    {
        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/provinces.json'), true) as $province) {
            $this->connection->insert(
                'pongho_provinces',
                $province,
                array(
                    'id'        => \PDO::PARAM_INT,
                    'region_id' => \PDO::PARAM_INT,
                    'name'      => \PDO::PARAM_STR,
                    'plate'     => \PDO::PARAM_STR,
                )
            );
        }
    }

    /**
     * Popola la tabella delle città.
     */
    protected function installCities()
    {
        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/cities.json'), true) as $city) {
            $this->connection->insert(
                'pongho_cities',
                $city,
                array(
                    'id'          => \PDO::PARAM_INT,
                    'province_id' => \PDO::PARAM_INT,
                    'region_id'   => \PDO::PARAM_INT,
                    'name'        => \PDO::PARAM_STR,
                )
            );
        }
    }

    /**
     * Popola la tabella delle traduzioni.
     */
    protected function installTranslations()
    {
        $field_key_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/translations.json'), true) as $translation) {
            $translation[$field_key_name] = $translation['key'];
            unset($translation['key']);

            $this->connection->insert(
                'pongho_translations',
                $translation,
                array(
                    'language_id'   => \PDO::PARAM_INT,
                    'application'   => \PDO::PARAM_STR,
                    'value'         => \PDO::PARAM_STR,
                    $field_key_name => \PDO::PARAM_STR,
                )
            );
        }
    }

    /**
     * Prepara le relazioni e i dati di base
     */
    protected function prepareBaseStuff()
    {
        $updates = array_unique(array_merge($this->getAvailableUpdates('Core'), $this->getAvailableUpdates('Admin')));
        sort($updates);

        $max = max($updates);

        $field_key_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $settings = array(
            array(
                $field_key_name => 'version',
                'value'         => '2.0.1825-alpha',
            ),
            array(
                $field_key_name => 'db_version',
                'value'         => $max,
            ),
            array(
                $field_key_name => 'installed_updates',
                'value'         => json_encode($updates),
            ),
        );

        foreach ($settings as $setting) {
            $this->connection->insert(
                'pongho_settings',
                $setting,
                array(
                    $field_key_name => \PDO::PARAM_STR,
                    'value'         => \PDO::PARAM_STR,
                )
            );
        }

        $site_id = reset($this->sites);

        // ModuleSite
        $this->connection->insert(
            'pongho_modules_sites',
            array(
                'module_id'  => $this->modules['user'],
                'site_id'    => $site_id,
                'is_enabled' => true,
            ),
            array(
                'module_id'  => \PDO::PARAM_INT,
                'site_id'    => \PDO::PARAM_INT,
                'is_enabled' => \PDO::PARAM_BOOL,
            )
        );

        // LanguageSite
        $this->connection->insert(
            'pongho_languages_sites',
            array(
                'site_id'     => $site_id,
                'language_id' => reset($this->languages),
                'path'        => '/',
                'is_enabled'  => true,
            ),
            array(
                'site_id'     => \PDO::PARAM_INT,
                'language_id' => \PDO::PARAM_INT,
                'path'        => \PDO::PARAM_STR,
                'is_enabled'  => \PDO::PARAM_BOOL,
            )
        );

        $language_site_id = $this->getLastInsertId('pongho_languages_sites');

        // LanguageModule
        $this->connection->insert(
            'pongho_languages_modules',
            array(
                'language_site_id' => $language_site_id,
                'module_id'        => $this->modules['user'],
                'path'             => '/user/',
                'name'             => 'Gestione utenti',
                'position'         => 10,
                'is_enabled'       => true,
            ),
            array(
                'language_site_id' => \PDO::PARAM_INT,
                'module_id'        => \PDO::PARAM_INT,
                'path'             => \PDO::PARAM_STR,
                'name'             => \PDO::PARAM_STR,
                'position'         => \PDO::PARAM_INT,
                'is_enabled'       => \PDO::PARAM_BOOL,
            )
        );

        // Applicazioni del sito
        foreach ($this->apps as $app_id) {
            $this->connection->insert(
                'pongho_apps_sites',
                array(
                    'app_id'  => $app_id,
                    'site_id' => $site_id,
                ),
                array(
                    'app_id'  => \PDO::PARAM_INT,
                    'site_id' => \PDO::PARAM_INT
                )
            );
        }

        $permits = array(
            'admin',
            'admin.options',
            'admin.options.translations',
            'admin.options.notifications',
            'users.admin',
            'users.add',
            'users.edit',
            'users.delete',
            'users.handle_groups',
        );

        // Permessi per l'amministratore
        foreach ($permits as $key) {
            $permit_id = $this->permits[$key];
            $this->connection->insert(
                'pongho_permits_roles',
                array(
                    'permit_id' => $permit_id,
                    'role_id'   => $this->roles['admin']
                ),
                array(
                    'permit_id' => \PDO::PARAM_INT,
                    'role_id'   => \PDO::PARAM_INT,
                )
            );
        }
    }

    /**
     * Restituisce l'id dell'ultimo record inserito, da usarsi appena dopo un inserimento e solo per tabelle che hanno
     * la chiave primaria "id"
     *
     * @param string $table_name
     * @return int
     */
    protected function getLastInsertId($table_name)
    {
        return $this->connection->lastInsertId($table_name . '_id_seq');
    }
}
