<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Form\Field;

use Application\Core\LinkType\LinkFacadeFactory;
use Pongho\Form\BaseField;

/**
 * Class AbstractLinkField
 */
abstract class AbstractLinkField extends BaseField
{
    protected static $json_printed = array();

    /**
     * @var \Application\Core\LinkType\LinkFacade
     */
    protected $facade;

    /**
     * {@inheritdoc}
     */
    public function getTag()
    {
        $facade = $this->getFacade();

        $code = $facade->render($this->getName(), $this->getValue());
        $code .= $this->getScriptCode();

        return $code;
    }

    /**
     * {@inheritdoc}
     */
    public function handle(array $request)
    {
        if ($this->isReadonly()) {
            return $this;
        }

        $facade = $this->getFacade();
        $link_type = $facade->save($request[$this->getKey()], $this->getValue(), $this->getLanguageId());
        $this->setValue($link_type ? $link_type->getId() : null);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getValue()
    {
        $value = parent::getValue();

        return is_numeric($value) ? $value : null;
    }

    /**
     * {@inheritdoc}
     */
    public function emptyValue()
    {
        return null;
    }

    /**
     * @return null|int
     */
    abstract protected function getLanguageId();

    /**
     * @return string
     */
    abstract protected function getCollectionName();

    /**
     * @return \Application\Core\LinkType\LinkFacade
     */
    protected function getFacade()
    {
        if ($this->facade === null) {
            if (!$this->hasSetting('facade_factory')) {
                throw new \InvalidArgumentException('The "facade_factory" setting is not defined for the LinkField "' . $this->getKey() . '"');
            }

            /** @var LinkFacadeFactory $facade_factory */
            $facade_factory = $this->getSetting('facade_factory');

            $this->facade = $facade_factory->create($this->getCollectionName(), $this->getLanguageId());
        }

        return $this->facade;
    }

    /**
     * @return string
     */
    protected function getScriptCode()
    {
        $facade = $this->getFacade();
        $code = '';
        $language_id = $this->getLanguageId();
        $coll_name = $this->getCollectionName();

        if (!isset(self::$json_printed[$coll_name][$language_id])) {
            if ($language_id) {
                $code = <<<HTML
Pongho.linkType["{$coll_name}"][{$language_id}] = {$facade->renderJson()};
HTML;
            } else {
                $code = <<<HTML
Pongho.linkType["{$coll_name}"]["null"] = {$facade->renderJson()};
HTML;
            }

            $code = <<<HTML
<script type="text/javascript">
var Pongho = Pongho || {};
Pongho.linkType = Pongho.linkType || {};
Pongho.linkType["{$coll_name}"] = Pongho.linkType["{$coll_name}"] || {};{$code}
</script>
HTML;

            self::$json_printed[$coll_name][$language_id] = true;
            return $code;
        }

        return $code;
    }
}
