<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core;

use Application\Core\Entity\AccountInterface;
use Application\Core\Exception\LoginInactiveUserException;
use Application\Core\Exception\LoginInvalidPasswordException;
use Application\Core\Exception\LoginRequiredPasswordException;
use Application\Core\Exception\LoginRequiredUsernameException;
use Application\Core\Utilities\UserFactoryInterface;
use Pongho\Http\Request;
use Pongho\Http\Session;

/**
 * Class Firewall
 */
class Firewall
{
    /**
     * @var Session
     */
    protected $session;

    /**
     * @var UserFactoryInterface
     */
    protected $user_factory;

    /**
     * @var User
     */
    protected $real_user;

    /**
     * @var User
     */
    protected $current_user;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @param Session              $session
     * @param Request              $request
     * @param UserFactoryInterface $user_factory
     */
    public function __construct(Session $session, Request $request, UserFactoryInterface $user_factory)
    {
        $this->session = $session;
        $this->request = $request;
        $this->user_factory = $user_factory;
    }

    /**
     * Impersona un diverso utente
     */
    protected function impersonateUser()
    {
        // Ho i permessi per impersonare un altro utente?
        if (!($this->getRealUser()->isFounder() || $this->getRealUser()->hasPermit('users.impersonation'))) {
            $this->stopImpersonation();
            return;
        }

        $real_user_id = (int) $this->session->get('user_id');
        $current_user_id = (int) $this->session->get('current_user_id');
        $switch_id = (int) $this->request->query->get('switch_user');

        // Evito di impersonare me stesso, più che altro per evitare di creare due istanze di User
        if ($switch_id === $real_user_id) {
            $this->stopImpersonation();
            return;
        }

        $user_id = null;

        // Ho salvato il current_user_id in sessione?
        if ($current_user_id && $current_user_id !== $real_user_id) {
            $user_id = $current_user_id;
        }

        // Sto passando un id in query string?
        if ($switch_id && $switch_id !== $current_user_id) {
            $user_id = $switch_id;
        }

        if ($user_id !== null) {
            if ($this->current_user === null) {
                $this->current_user = $this->user_factory->getUser($user_id);
            }

            $this->session->set('current_user_id', $user_id);
        }
    }

    /**
     * Esce dall'impersonazione dell'utente
     */
    protected function stopImpersonation()
    {
        $this->session->del('current_user_id');
        $this->current_user = null;
    }

    /**
     * @return User
     */
    public function getCurrentUser()
    {
        // Impersono un utente
        if ($this->request->query->has('switch_user') || $this->session->has('current_user_id')) {
            $this->impersonateUser();
        }

        // Esco dall'impersonazione
        if ($this->request->query->has('stop_impersonation')) {
            $this->stopImpersonation();
        }

        // Non sto impersonando nessuno
        if ($this->current_user === null) {
            return $this->getRealUser();
        }

        return $this->current_user;
    }

    /**
     * @return Session
     */
    public function getSession()
    {
        return $this->session;
    }

    /**
     * Prepara l'utente reale
     */
    protected function prepare()
    {
        if (!$this->real_user) {
            $user_id = null;

            if ($this->getSession()->has('user_id')) {
                $user_id = (int) $this->getSession()->get('user_id');
            }

            $this->real_user = $this->user_factory->getUser($user_id);
        }
    }

    /**
     * @return User
     */
    public function getRealUser()
    {
        $this->prepare();

        return $this->real_user;
    }

    /**
     * Effettua il login
     *
     * @param $username_or_email
     * @param $password
     * @return bool
     */
    public function login($username_or_email, $password)
    {
        // Campo nome utente o email obbligatorio
        if (empty($username_or_email)) {
            throw new LoginRequiredUsernameException();
        }

        // Campo password obbligatorio
        if (empty($password)) {
            throw new LoginRequiredPasswordException();
        }

        $account = $this->user_factory->getAccount($username_or_email);

        // Confronto la password dell'account con quella passata
        if ($account->getPassword() !== $password) {
            throw new LoginInvalidPasswordException();
        }

        // Utente attivo?
        if (!$account->isActive()) {
            throw new LoginInactiveUserException();
        }

        $this->real_user->forceLogin($account);

        $this->getSession()->set('user_id', $account->getId());

        return true;
    }

    /**
     * @param AccountInterface $account
     */
    public function forceLogin(AccountInterface $account)
    {
        $this->getRealUser()->forceLogin($account);

        $this->getSession()->set('user_id', $account->getId());
    }

    /**
     * Effettua il logout
     */
    public function logout()
    {
        $this->getRealUser()->logout();

        $this->getSession()->destroy();
    }
}
