<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Entity;

use Pongho\Utilities\DateTime;

/**
 * Interface AccountInterface
 *
 * @todo: dividere in account e anagrafica
 */
interface AccountInterface
{
    /**
     * @return int
     */
    public function getId();

    /**
     * @return LanguageInterface
     */
    public function getLanguage();

    /**
     * @return string
     */
    public function getUsername();

    /**
     * @throw \InvalidArgumentException
     * @param string $username
     */
    public function setUsername($username);

    /**
     * @return string
     */
    public function getEmail();

    /**
     * @throw \InvalidArgumentException
     * @param string $email
     */
    public function setEmail($email);

    /**
     * Restituisce la versione criptata della password
     *
     * @return string
     */
    public function getPassword();

    /**
     * @throw \InvalidArgumentException
     * @param string $password
     */
    public function setPassword($password);

    /**
     * @return DateTime
     */
    public function getSubscribeDate();

    /**
     * @return DateTime
     */
    public function getLastUpdateDate();

    /**
     * Elimina l’avatar utente.
     *
     * @return boolean
     */
    public function deleteAvatar();

    /**
     * Restituisce il modello dell’avatar.
     *
     * @return ImageInterface
     */
    public function getAvatar();

    /**
     * Restituisce la chiave di attivazione dell'utente
     *
     * @return string
     */
    public function getActivationKey();

    /**
     * Imposta la chiave di attivazione per l'utente
     *
     * @return $this
     */
    public function resetActivationKey();

    /**
     * Indica se l'utente è attivo, non attivo o disabilitato dall'amministratore
     *
     * @return bool|null
     */
    public function isActive();

    /**
     * @return bool
     */
    public function isAnonymous();

    /**
     * @return bool
     */
    public function isAdmin();

    /**
     * @return bool
     */
    public function isFounder();

    /**
     * @return bool
     */
    public function isBusiness();

    /**
     * @param $key
     * @return bool
     */
    public function hasPermit($key);

    /**
     * @return bool
     */
    public function isDeletable();

    /**
     * Restituisce i genitori diretti dell’utente.
     *
     * @return \Application\Core\Model\Account[]
     *
     * @api
     */
    public function getParents();

    /**
     * Imposta i genitori diretti dell’utente.
     *
     * @param AccountInterface[] $parents
     * @return $this
     *
     * @api
     */
    public function setParents(array $parents);

    /**
     * Imposta un genitore diretto dell’utente.
     *
     * @param int|AccountInterface $parent_id_or_parent_instance
     * @throws \LogicException
     * @return $this
     *
     * @api
     */
    public function setParent($parent_id_or_parent_instance);

    /**
     * Rimuove un genitore diretto dell’utente.
     *
     * @param int|AccountInterface $parent_id_or_instance
     * @return bool
     *
     * @api
     */
    public function delParent($parent_id_or_instance);

    /**
     * Indica se l’utente ha un genitore diretto.
     *
     * @param int $parent_id L'ID del genitore che si vuole controllare.
     * @return bool
     *
     * @api
     */
    public function hasParent($parent_id);

    /**
     * Indica se l'utente ha un certo antenato
     *
     * @param $ancestor_id
     * @return bool
     */
    public function hasAncestor($ancestor_id);

    /**
     * Restituisce i discendenti.
     *
     * @return AccountInterface[]
     */
    public function getDescendants();

    /**
     * Restituisce il numero di discendenti.
     *
     * @return int
     */
    public function countDescendants();

    /**
     * Indica se l'utente ha un certo discendente
     *
     * @param $descendant_id
     * @return bool
     */
    public function hasDescendant($descendant_id);

    /**
     * Genera l'url alla creazione dell'utente.
     */
    public function generateUrl();

    /**
     * Aggiunge la relazione a se stesso nella tabella del portfolio.
     *
     * @ignore
     */
    public function addSelfPortfolioRelation();

    /**
     * Elimina le relazioni dell’utente e sposta tutti i suoi figli sotto il controllo di tutti i suoi padri.
     *
     * @return bool
     *
     * @ignore
     */
    public function deletePortfolioRelations();

    /**
     * Attiva l'account
     *
     * @return $this
     */
    public function activate();

    /**
     * Disattiva l'account
     *
     * @return $this
     */
    public function deactivate();

    /**
     * Restituisce il nome completo dell’utente.
     * Se il nome risulta vuoto verrà utilizzato lo username.
     * È possibile modificare il comportamento di questa funzione tramite l'evento model.pongho_users.name_function.
     *
     * Esempio:
     * <code>
     *   $account->name = 'Paolino';
     *   $account->surname = 'Paperino';
     *   echo $account->name(); // Paolino Paperino
     *   echo $account->name(true); // Paperino Paolino
     * </code>
     *
     * @param boolean $invert Se invertire l’ordine di nome e cognome.
     * @return string
     *
     * @todo spostare nell'interfaccia dell'anagrafica
     */
    public function name($invert = false);
}
