<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Firewall;
use Application\Core\User;
use Application\Core\Utilities\UserFactory;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Pongho\Http\Session;

/**
 * Class UserServiceProvider
 */
class UserServiceProvider implements ServiceProviderInterface
{
    protected static $shared = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('user_factory', array($this, 'getUserFactoryService'));
        $container->setService('firewall', array($this, 'getFirewallService'));
        $container->setService('session', array($this, 'getSessionService'));
        $container->setService('current_user', array($this, 'getCurrentUserService'));
    }

    /**
     * @return UserFactory
     */
    public function getUserFactoryService()
    {
        if (!array_key_exists('user_factory', self::$shared)) {
            self::$shared['user_factory'] = new UserFactory();
        }

        return self::$shared['user_factory'];
    }

    /**
     * Restituisce il servizio del firewall.
     *
     * @param Container $container
     * @return Firewall
     */
    public function getFirewallService(Container $container)
    {
        if (!array_key_exists('firewall', self::$shared)) {
            self::$shared['firewall'] = new Firewall(
                $container->getService('session'),
                $container->getService('request'),
                $container->getService('user_factory')
            );
        }

        return self::$shared['firewall'];
    }

    /**
     * Restituisce l’utente corrente.
     *
     * @access protected
     * @param Container $container
     * @return User
     */
    public function getCurrentUserService(Container $container)
    {
        return $container->getService('firewall')->getCurrentUser();
    }

    /**
     * Restituisce la sessione.
     *
     * @return \Pongho\Http\Session
     */
    public function getSessionService()
    {
        if (!array_key_exists('session', self::$shared)) {
            self::$shared['session'] = Session::instance();
        }

        return self::$shared['session'];
    }
}
