<?php

/**
 * Questo file è parte di Pongho (Dev).
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Command\TranslationDebugCommand;
use Application\Core\Command\TranslationUpdateCommand;
use Application\Core\I18n\Translator\Translator;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Symfony\Component\Translation\Loader\ArrayLoader;
use Symfony\Component\Translation\Loader\XliffFileLoader;

/**
 * TranslationServiceProvider
 */
class TranslationServiceProvider implements ServiceProviderInterface
{
    /**
     * @var array
     */
    protected static $sharedServices = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('translator', array($this, 'getTranslatorService'));

        $container->extend(
            'console_commands',
            function (\ArrayObject $commands, Container $container) {
                $commands->append(new TranslationDebugCommand($container->getService('translator')));
                $commands->append(new TranslationUpdateCommand());
            }
        );
    }

    /**
     * @param Container $container
     * @return Translator
     */
    public function getTranslatorService(Container $container)
    {
        if (!array_key_exists('translator', static::$sharedServices)) {
            /** @var \Application\Core\User $user */
            $user = $container->getService('current_user');
            $culture = $user->getAccount()->language->culture;

            $translator = new Translator($culture);

            $translator->addLoader('xlffile', new XliffFileLoader());
            $translator->addLoader('localization', new ArrayLoader());
            $translator->setFallbackLocales(array('it-IT'));

            $translator->addApplicationTranslations('Core', $culture);

            /** @var \Application\Core\Localization $lang */
            $lang = $container->getService('language');
            $translator->addResource('localization', $lang->all(), $culture, 'lang');

            static::$sharedServices['translator'] = $translator;
        }

        return static::$sharedServices['translator'];
    }
}
