<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Template\TemplateHelperBuilder;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class TemplatingServiceProvider
 */
class TemplatingServiceProvider implements ServiceProviderInterface
{
    protected static $shared = array();

    /**
     * Permette di registrare dei servizi aggiuntivi al contenitore che gli viene passato.
     *
     * Questo metodo dovrebbe solo impostare o estendere dei servizi,
     * ma non dovrebbe mai richiederli.
     *
     * @param Container $container
     */
    public function register(Container $container)
    {
        $container->extend('template_engine', array($this, 'extendTemplateEngineService'));
        $container->setService('theme_view', array($this, 'getThemeViewService'));
        $container->setService('global_view_vars', array($this, 'getGlobalViewVarsService'));

        $container->setService('template_helper_builder', array($this, 'getTemplateHelperBuilderService'));
        $container->setService('template_head', array($this, 'getTemplateHeaderService'));
        $container->setService('template_footer', array($this, 'getTemplateFooterService'));
    }

    /**
     * @param Container $container
     * @deprecated Template\Engine non viene più usato, resta però per retrocompatibilità
     */
    public function extendTemplateEngineService(Container $container)
    {
        $container->getService('template_engine')->assignVars($container->getService('global_view_vars'));
    }

    /**
     * Restituisce il tema della vista.
     *
     * @param Container $container
     * @return \Pongho\Template\Theme Il tema della vista.
     */
    public function getThemeViewService(Container $container)
    {
        if (!array_key_exists('theme_view', self::$shared)) {
            $class = $container->getParameter('template.theme_class');

            /** @var \Application\Core\Model\Site $site */
            $site = $container->getService('site');

            /** @var \Pongho\Template\Theme $theme */
            $theme = new $class($site->getThemesPath());

            $theme->assignVars($container->getService('global_view_vars'));

            self::$shared['theme_view'] = $theme;
        }

        return self::$shared['theme_view'];
    }

    /**
     * Restituisce l’elenco delle variabili globali da impostare nelle viste.
     *
     * @param Container $container
     * @return array
     */
    public function getGlobalViewVarsService(Container $container)
    {
        return array(
            'head'     => $container->getService('template_head'),
            'footer'   => $container->getService('template_footer'),
            'html'     => $container->getService('template_html'),
            'lang'     => $container->getService('language'),
            'language' => $container->getService('language')->getLanguage(),
            'user'     => $container->getService('current_user'),
            'site'     => $container->getService('site'),
        );
    }

    /**
     * @param Container $container
     * @return TemplateHelperBuilder
     */
    public function getTemplateHelperBuilderService(Container $container)
    {
        if (!array_key_exists('template_helper_builder', self::$shared)) {
            $builder = new TemplateHelperBuilder(
                $container->getService('site'),
                $container->getService('language_site')->getLanguage()
            );

            self::$shared['template_helper_builder'] = $builder;
        }

        return self::$shared['template_helper_builder'];
    }

    /**
     * @param Container $container
     * @return \Pongho\Template\HeadHelper
     */
    public function getTemplateHeaderService(Container $container)
    {
        if (!array_key_exists('template_head', self::$shared)) {
            /** @var TemplateHelperBuilder $builder */
            $builder = $container->getService('template_helper_builder');

            self::$shared['template_head'] = $builder->buildHeader();
        }

        return self::$shared['template_head'];
    }

    /**
     * @param Container $container
     * @return \Pongho\Template\FooterHelper
     */
    public function getTemplateFooterService(Container $container)
    {
        if (!array_key_exists('template_footer', self::$shared)) {
            /** @var TemplateHelperBuilder $builder */
            $builder = $container->getService('template_helper_builder');

            self::$shared['template_footer'] = $builder->buildFooter();
        }

        return self::$shared['template_footer'];
    }
}
