<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Localization;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\Manager\ApplicationManager;
use Application\Core\Model\Manager\CoreManagerRegistry;
use Application\Core\Model\Manager\CoreManagerRegistryCache;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Model\Manager\LanguageSiteManager;
use Application\Core\Model\Manager\LanguageModuleManager;
use Application\Core\Model\Manager\ModuleManager;
use Application\Core\Model\Manager\ModuleSiteManager;
use Application\Core\Model\Manager\NotificationTemplateManager;
use Application\Core\Model\Manager\SettingManager;
use Application\Core\Model\Manager\SiteManager;
use Application\Core\Model\Site;
use Application\Core\User;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class ModelManagerProvider
 */
class ModelManagerProvider implements ServiceProviderInterface
{
    /**
     * @var array
     */
    protected static $shared = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('application_manager', array($this, 'getApplicationManagerService'));
        $container->setService('setting_manager', array($this, 'getSettingManagerService'));

        // Manager dei modelli base
        $container->setService('core_manager_registry', array($this, 'getCoreManagerRegistryService'));
        $container->setService('site_manager', array($this, 'getSiteManagerService'));
        $container->setService('language_manager', array($this, 'getLanguageManagerService'));
        $container->setService('module_manager', array($this, 'getModuleManagerService'));
        $container->setService('module_site_manager', array($this, 'getModuleSiteManagerService'));
        $container->setService('language_site_manager', array($this, 'getLanguageSiteManagerService'));
        $container->setService('language_module_manager', array($this, 'getLanguageModuleManagerService'));

        $container->setService('site', array($this, 'getSiteService'));
        $container->setService('main_site', array($this, 'getMainSiteService'));
        $container->setService('notification_template_manager', array($this, 'getNotificationTemplateManager'));

        $container->setService('language_site', array($this, 'getLanguageSiteService'));
        $container->setService('language', array($this, 'getLanguageService'));
    }

    /**
     * @param Container $container
     * @return ApplicationManager
     */
    public function getApplicationManagerService(Container $container)
    {
        if (!array_key_exists('application_manager', self::$shared)) {
            self::$shared['application_manager'] = new ApplicationManager($container->getService('site'));
        }

        return self::$shared['application_manager'];
    }

    /**
     * @return \Application\Core\Model\Manager\SettingManager
     */
    public function getSettingManagerService()
    {
        if (!array_key_exists('setting_manager', self::$shared)) {
            self::$shared['setting_manager'] = new SettingManager();
        }

        return self::$shared['setting_manager'];
    }

    /**
     * @return CoreManagerRegistry
     */
    public function getCoreManagerRegistryService()
    {
        if (!array_key_exists('core_manager_registry', self::$shared)) {
            self::$shared['core_manager_registry'] = new CoreManagerRegistry(new CoreManagerRegistryCache());
        }

        return self::$shared['core_manager_registry'];
    }

    /**
     * @param Container $container
     * @return SiteManager
     */
    public function getSiteManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getSiteManager();
    }

    /**
     * @param Container $container
     * @return LanguageManager
     */
    public function getLanguageManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getLanguageManager();
    }

    /**
     * @param Container $container
     * @return ModuleManager
     */
    public function getModuleManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getModuleManager();
    }

    /**
     * @param Container $container
     * @return ModuleSiteManager
     */
    public function getModuleSiteManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getModuleSiteManager();
    }

    /**
     * @param Container $container
     * @return LanguageSiteManager
     */
    public function getLanguageSiteManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getLanguageSiteManager();
    }

    /**
     * @param Container $container
     * @return LanguageModuleManager
     */
    public function getLanguageModuleManagerService(Container $container)
    {
        return $container->getService('core_manager_registry')->getLanguageModuleManager();
    }

    /**
     * Restituisce la lingua.
     *
     * @access protected
     * @param Container $container
     * @return \Application\Core\Model\Language|\Application\Core\Model\LanguageSite
     */
    public function getLanguageSiteService(Container $container)
    {
        if (!array_key_exists('language_site', self::$shared)) {
            /**
             * La lingua può essere decisa in base a 3 parametri:
             *
             * 1. l’indirizzo della pagina
             * 2. le impostazioni dell'utente corrente
             * 3. la lingua impostata nel browser
             *
             * Al terzo parametro ci si arriva solo se l’utente è anonimo. In questo caso, devo controllare
             * se la lingua del browser è disponibile. In alternativa si utilizza quella di default del sito.
             */

            // primo caso, controllo se la rotta ha impostato una lingua (salto se sono in amministrazione o nelle api)
            $request = $container->getService('request');
            if (substr($request->getPathInfo(), 0, 8) !== '/pongho/') {
                $router = $container->getService('router');
                $parameters = $router->getParameters();

                if (array_key_exists('site_language', $parameters) && $parameters['site_language'] instanceof LanguageSite) {
                    self::$shared['language_site'] = $parameters['site_language'];

                    return self::$shared['language_site'];
                }
            }

            /**
             * @var User $user
             * @var Site $site
             * @var LanguageSiteManager $language_site_manager
             * @var LanguageSite $language_site
             */

            // secondo e terzo caso, vediamo l’utente
            $user = $container->getService('current_user');
            $site = $container->getService('site');
            $language_site_manager = $container->getService('language_site_manager');
            $language_site = $language_site_manager->findByLanguageAndSite($user->language_id, $site->id);

            if ($user->isLogged() && ($language_site !== null)) {
                // secondo caso
                self::$shared['language_site'] = $language_site;
            } else {
                // terzo caso
                $request = $container->getService('request');

                $languages = array();
                foreach ($site->getLanguages() as $language_site) {
                    $languages[$language_site->getLanguage()->getIso()] = $language_site;
                }

                $preferred = $request->getPreferredLanguage(array_keys($languages));

                if ($preferred && array_key_exists($preferred, $languages)) {
                    // lingua dal browser
                    self::$shared['language_site'] = $languages[$preferred];
                } else {
                    // lingua di default del sito
                    self::$shared['language_site'] = $site->getDefaultLanguage();
                }
            }
        }

        return self::$shared['language_site'];
    }

    /**
     * Restituisce il servizio di traduzione
     *
     * @param Container $container
     * @return \Application\Core\Localization
     */
    public function getLanguageService(Container $container)
    {
        if (!array_key_exists('language', self::$shared)) {
            /** @var \Application\Core\Model\LanguageSite|\Application\Core\Model\Language $language_site */
            $language_site = $container->getService('language_site');

            self::$shared['language'] = new Localization($language_site);
        }

        return self::$shared['language'];
    }

    /**
     * Restituisce il modello del sito corrente.
     *
     * @param Container $container
     * @return \Application\Core\Model\Site
     */
    public function getSiteService(Container $container)
    {
        if (!array_key_exists('site', self::$shared)) {
            $site_id = $container->getParameter('site_id');

            /** @var \Pongho\Http\Request $request */
            $request = $container->getService('request');

            if (substr($request->getPathInfo(), 0, 8) === '/pongho/' && $request->query->has('site')) {
                $site_id = $request->query->get('site');
            }

            if ((self::$shared['site'] = $container->getService('site_manager')->findById($site_id)) === null) {
                throw new \RuntimeException('Site not found!');
            }
        }

        return self::$shared['site'];
    }

    /**
     * Restituisce il modello del sito principale.
     *
     * @access protected
     * @param Container $container
     * @return \Application\Core\Model\Site
     */
    public function getMainSiteService(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        if (!array_key_exists('main_site', self::$shared)) {
            try {
                $site = $container->getService('site');
            } catch (\RuntimeException $e) {
                // Se non trovo il sito corrente, converto il messaggio dell’eccezione
                throw new \RuntimeException('Main site not found!');
            }

            if ($site->getId() !== 1) {
                if (($site = $container->getService('site_manager')->findById(1)) === null) {
                    throw new \RuntimeException('Main site not found!');
                }
            }

            self::$shared['main_site'] = $site;
        }

        return self::$shared['main_site'];
    }

    /**
     * @param Container $container
     * @return NotificationTemplateManager
     */
    public function getNotificationTemplateManager(Container $container)
    {
        if (!array_key_exists('notification_template_manager', self::$shared)) {
            self::$shared['notification_template_manager'] = new NotificationTemplateManager(
                $container->getService('site'),
                $container->getService('notification_collection')
            );
        }

        return self::$shared['notification_template_manager'];
    }
}
