<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Mailer\MailerHelper;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class MailerServiceProvider
 */
class MailerServiceProvider implements ServiceProviderInterface
{
    protected static $shared = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('mailer', array($this, 'getMailerService'));
        $container->setService('mailer_transport', array($this, 'getMailerTransportService'));
        $container->setService('mailer_helper', array($this, 'getMailerHelperService'));
    }

    /**
     * Restituisce il gestore delle email.
     *
     * @param Container $container
     * @return \Application\Core\Utilities\Mailer
     */
    public function getMailerService(Container $container)
    {
        if ( !array_key_exists('mailer', self::$shared) )
        {
            $mailer_class = $container->getParameter('mailer.mailer_class');

            /** @var \Application\Core\Utilities\Mailer $mailer */
            $mailer = new $mailer_class($container->getService('site'), $container->getService('language'));
            $mailer->setTransport($container->getService('mailer_transport'));

            self::$shared['mailer'] = $mailer;
        }

        return self::$shared['mailer'];
    }

    /**
     * Restituisce il gestore delle email.
     *
     * @param Container $container
     * @return \Swift_Transport
     */
    public function getMailerTransportService(Container $container)
    {
        if ( !array_key_exists('mailer_transport', self::$shared) )
        {
            if ($container->hasParameter('mailer.transport')) {
                $transport = $container->getParameter('mailer.transport');

                if (!is_callable($transport)) {
                    throw new \LogicException('Parameter "mailer.transport" must be callable!');
                }

                /** @var \Swift_Transport $transport */
                $transport = call_user_func($transport);
            } else {
                $class = $container->getParameter('mailer.transport_class');
                $params = $container->getParameter('mailer.transport_params');

                /** @var \Swift_Transport $transport */
                $transport = call_user_func_array(array($class, 'newInstance'), $params);
            }

            self::$shared['mailer_transport'] = $transport;
        }

        return self::$shared['mailer_transport'];
    }

    /**
     * @param Container $container
     * @return MailerHelper
     */
    public function getMailerHelperService(Container $container)
    {
        if (!array_key_exists('mailer_helper', self::$shared)) {
            self::$shared['mailer_helper'] = new MailerHelper(
                $container->getService('site'),
                $container->getService('translator')
            );
        }

        return self::$shared['mailer_helper'];
    }
}
