<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Monolog\Formatter\HtmlFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SwiftMailerHandler;
use Monolog\Logger;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class LoggerServiceProvider
 */
class LoggerServiceProvider implements ServiceProviderInterface
{
    protected static $shared = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('logger_message', array($this, 'getLoggerMessageService'));
        $container->setService('logger', array($this, 'getLoggerService'));
    }

    /**
     * Restituisce il messaggio email per gli errori critici dei log
     *
     * @param Container $container
     * @return \Swift_Message
     */
    public function getLoggerMessageService(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $container->getService('site');

        $message = new \Swift_Message();

        $message
            ->addFrom($container->getParameter('logger.mail.from', 'noreply@metaline.it'))
            ->addTo($container->getParameter('logger.mail.to', 'log@metaline.it'))
            ->setSubject($site->getName())
            ->setContentType('text/html');

        return $message;
    }

    /**
     * Restituisce il gestore dei log
     *
     * @param Container $container
     * @return \Monolog\Logger
     */
    public function getLoggerService(Container $container)
    {
        if (!array_key_exists('logger', self::$shared)) {
            $logger_name = 'pongho';
            $logger = new Logger($logger_name);

            /** @var \Application\Core\Model\Site $site */
            $site = $container->getService('site');

            // Salvo le informazioni di log su file
            $logger->pushHandler(
                new RotatingFileHandler(
                    $site->getContentPath(
                        $container->getParameter('logger.path', '/log') . '/' . $logger_name . '.log'
                    ),
                    $container->getParameter('logger.maxfiles', 90),
                    $container->getParameter(
                        'logger.file.level',
                        $container->getService('kernel')->isDebug() ? Logger::DEBUG : Logger::INFO
                    )
                )
            );

            // Mail per gli errori critici
            $swift_mailer_handler = new SwiftMailerHandler(
                new \Swift_Mailer($container->getService('mailer_transport')),
                $container->getService('logger_message'),
                Logger::DEBUG
            );

            $swift_mailer_handler->setFormatter(new HtmlFormatter());

            // Mail per gli errori critici
            $logger->pushHandler(
                new FingersCrossedHandler(
                    new BufferHandler(
                        $swift_mailer_handler
                    ), Logger::ERROR
                )
            );

            self::$shared['logger'] = $logger;
        }

        return self::$shared['logger'];
    }
}
