<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use ActiveRecord\CacheWrapper;
use ActiveRecord\Config;
use ActiveRecord\Connection;
use ActiveRecord\Profiler\Profiler;
use Doctrine\DBAL\Configuration;
use Doctrine\DBAL\DriverManager;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class DbServiceProvider
 */
class DbServiceProvider implements ServiceProviderInterface
{
    protected static $shared = array();

    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->setService('db_profiler', array($this, 'getDbProfilerService'));
        $container->setService('connection', array($this, 'getConnectionService'));
        $container->setService('connection_schema', array($this, 'getConnectionSchemaService'));

        // Non sarebbe affar suo ma è per tenere tutto quello relativo al database nello stesso posto
        $this->registerOrm($container);
    }


    /**
     * Restituisce il profilatore delle query in database.
     *
     * @return \ActiveRecord\Profiler\ProfilerInterface
     */
    public function getDbProfilerService()
    {
        if (!array_key_exists('db_profiler', self::$shared)) {
            self::$shared['db_profiler'] = new Profiler();
        }

        return self::$shared['db_profiler'];
    }

    /**
     * Restituisce la connessione (DBAL) al database.
     *
     * @return \Doctrine\DBAL\Connection
     */
    public function getConnectionService()
    {
        if (!array_key_exists('connection', self::$shared)) {
            $pdo = Connection::instance()->connection;
            $config = new Configuration();

            $connectionParams = array(
                'pdo' => $pdo,
            );

            self::$shared['connection'] = DriverManager::getConnection($connectionParams, $config);
        }

        return self::$shared['connection'];
    }

    /**
     * Restituisce lo schema (DBAL) del database.
     *
     * @param Container $container
     * @return \Doctrine\DBAL\Schema\Schema
     */
    public function getConnectionSchemaService(Container $container)
    {
        if (!array_key_exists('connection_schema', self::$shared)) {
            /** @var \Doctrine\DBAL\Connection $conn */
            $conn = $container->getService('connection');
            $sm = $conn->getSchemaManager();

            self::$shared['connection_schema'] = $sm->createSchema();
        }

        return self::$shared['connection_schema'];
    }

    /**
     * Imposta la configurazione per la connessione al database.
     *
     * @param Container $container
     */
    protected function registerOrm(Container $container)
    {
        $path = $container->getParameter('orm.config_path');

        if (!file_exists($path)) {
            throw new \UnexpectedValueException(sprintf('Database configuration file not exists on path "%s"!', $path));
        }

        $settings = include $path;

        if (!array_key_exists('connections', $settings)) {
            throw new \UnexpectedValueException(sprintf('Connections not found on %s file.', $path));
        }

        if (!array_key_exists('default_connection', $settings)) {
            throw new \UnexpectedValueException(sprintf('Default connection not found on %s file.', $path));
        }

        $default_connection = $settings['default_connection'];

        if ($container->getService('kernel')->getEnvironment() === 'test' && isset($settings['connections']['tests'])) {
            $default_connection = 'tests';
        }

        Config::instance()
            ->setConnections($settings['connections'])
            ->setDefaultConnectionName($default_connection)
            ->setDebug($container->getService('kernel')->isDebug())
            ->setEventDispatcher($container->getService('event_dispatcher'));

        if ($container->hasService('db_profiler') && $container->getService('kernel')->isDebug()) {
            Config::instance()->setProfiler($container->getService('db_profiler'));
        }

        try {
            Config::instance()->setCache(new CacheWrapper($container->getService('cache')));
        } catch (\BadMethodCallException $e) {
            // do nothing
        }
    }
}
