<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\AdminController;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\Generator\AdminGenerator;
use Application\Cms\Form\FormBuilder;
use Application\Core\Utilities\UsersImportHelper;
use Pongho\Form\Subject\ArrayRowSubject;
use Pongho\Form\Table\TableRow;

/**
 * Class UsersImportController
 */
class UsersImportController extends AdminController
{
    /*
     * Nuove specifiche:
     * - file CSV per ridurre i possibili errori di lettura nel caso l'xml non sia correttamente formattato
     *
     * Nuova procedura di importazione:
     * - carico il file, viene salvato temporaneamente (come succede ora)
     * - mostra la schermata di selezione delle colonne, caricando per anteprima solo una decina di righe,
     *   qui è presente una form o un campo nascosto per memorizzare il nome del file, che potrebbe anche essere sempre
     *   lo stesso
     * - iniziando l'importazione verrà riletto il file precedentemente salvato e importato nella sua interezza,
     *   eventuali errori di importazione quali indirizzi email malformati, dati duplicati (username) o errori di
     *   validazione/salvataggio verranno notificati.
     * - in caso di errori bisogna correggere il file CSV e reimportarlo, la reimportazione non modificherà alcun dato
     *   già esistente in database ma importerà solo i nuovi nominativi corretti
     */

    /**
     * @var UsersImportHelper
     */
    protected $import_helper;

    /**
     * Primo passaggio: Caricamento del file da importare
     */
    public function indexAction()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $this->cleanUp();

        /** @var \Pongho\Form\Form $form */
        $form = FormBuilder::buildConfig($this->getUploadFormConfig());
        $form->setGenerator(new AdminGenerator($translator));
        $form->handleRequest($this->getRequest());

        if ($form->hasHandled()) {
            /** @var \ArrayObject $model */
            $model = $form->getSubject()->getModel();
            $this->getSession()
                ->set('import-config', $model->getArrayCopy());

            return $this->getHelper()->redirectResponse(pongho_url($this->getHelper()->getPath() . 'import/'));
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(PONGHO_PATH . '/Application/Admin/Resources/views/edit.php')
            ->assignVars(
                array(
                    'title'   => $translator->trans('Users import'),
                    'content' => $form,
                )
            );

        return null;
    }

    /**
     * Secondo pssaggio: Importazione effettiva dei dati
     */
    public function importAction()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');
        $this->getImportHelper()->readFile();

        /** @var \Pongho\Form\Form $form */
        $form = FormBuilder::buildConfig($this->getImportFormConfig());
        $form->setGenerator(new AdminGenerator($translator));
        $form->handleRequest($this->getRequest());

        if ($form->hasHandled()) {
            // Rileggo tutto il file e associo le colonne che mi sono arrivate nella prima riga del Subject
            /** @var ArrayRowSubject[] $table */
            $table = $form->getSubject()->get('table');
            if (!isset($table[0])) {
                throw new \RuntimeException('The file must be empty, no headers found');
            }

            $result = $this->getImportHelper()->createAccounts($table[0]->getModel()->getArrayCopy());

            $this->getSession()
                ->set('import-result', $result);

            return $this->getHelper()->redirectResponse(pongho_url($this->getHelper()->getPath() . 'result/'));
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(PONGHO_PATH . '/Application/Admin/Resources/views/edit.php')
            ->assignVars(
                array(
                    'title'   => $translator->trans('Users import - step 2'),
                    'content' => $form,
                )
            );

        return null;
    }

    /**
     * Step finale dell'importazione
     */
    public function resultAction()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($this->getSession()->has('import-result')) {
            $result = $this->getSession()->get('import-result');
            $count = $result['import_count'];

            if ($result['errors']) {
                $error_table = '';
                foreach ($result['errors'] as $error) {
                    $error_table .= '<tr><td class="smallest">' . $error[0] . '</td><td class="main">' . $error[1] . '</td><td>' . $error[2] . '</td></tr>';
                }

                // Messaggio
                $content = '<p>' . $translator->trans(
                        'The import has been completed. There were some errors. %users% new users were imported and the next records need to be fixed before being imported correctly:',
                        array('%users%' => $count)
                    ) . '</p>';

                // Link per scaricare il file contenente le righe che hanno generato errore
                $url = $this->getHelper()->getSite()->getThemesUrl('/../temp/users-import-errors.csv');
                $content .= sprintf(
                    '<p><a href="%s">%s</a></p>',
                    $url,
                    $translator->trans('Download the file with the errors')
                );

                // Link per ritornare all'inizio dell'importazione
                $content .= sprintf(
                    '<p>%s</p>',
                    $translator->trans(
                        'Please, revise the errors and <a href="%url%">restart the process</a>',
                        array('%url%' => pongho_url($this->getHelper()->getPath()))
                    )
                );

                // Tabella errori
                $content .= sprintf(
                    '<table class="pongho-table"><thead><tr><th class="smallest">%s</th><th class="main">%s</th><th class="smallest">%s</th></tr></thead>%s</table>',
                    $translator->trans('Row'),
                    $translator->trans('Message'),
                    $translator->trans('Value'),
                    $error_table
                );
            } else {
                $content = '<p>' . $translator->trans(
                        'The import has been completed correctly. %users% new users were imported.',
                        array('%users%' => $count)
                    ) . '</p>';
            }
        } else {
            $content = '<p>' . $translator->trans(
                    'Something went wrong, try <a href="%url%">restarting the import</a> process.',
                    array('%url%' => pongho_url($this->getHelper()->getPath()))
                ) . '</p>';
        }

        $this->cleanUp();

        $this->getHelper()->getBodyView()
            ->setTemplatePath(PONGHO_PATH . '/Application/Admin/Resources/views/edit.php')
            ->assignVars(
                array(
                    'title'   => $translator->trans('Users import - result'),
                    'content' => '<div class="panel">' . $content . '</div>',
                )
            );
    }

    /**
     * Pulisce la sessione e i file creati (tranne quello degli errori che deve restare accessibile)
     */
    protected function cleanUp()
    {
        // Faccio un po' di pulizia
        if ($this->getSession()->has('import-config')) {
            $this->getSession()->del('import-config');
        }

        if ($this->getSession()->has('import-result')) {
            $this->getSession()->del('import-result');
        }

        $file = $this->getImportHelper()->getUploadPath() . '/' . $this->getImportHelper()->getFilename();
        if (file_exists($file)) {
            unlink($file);
        }
    }

    /**
     * @return UsersImportHelper
     */
    protected function getImportHelper()
    {
        if ($this->import_helper === null) {
            $upload_path = $this->getHelper()->getSite()->getContentPath('/temp/');
            $this->import_helper = new UsersImportHelper($upload_path, $this->getContainer()->getService('translator'));
        }

        if ($this->getSession()->has('import-config')) {
            $config = $this->getSession()->get('import-config');

            $this->import_helper->setDelimiter($config['delimiter']);
            $this->import_helper->setEnclosure($config['enclosure']);
            $this->import_helper->setEscape($config['escape']);
        }

        return $this->import_helper;
    }

    /**
     * @return FormConfig
     */
    protected function getUploadFormConfig()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');
        $subject = new ArrayRowSubject(array('delimiter' => ';', 'enclosure' => '"', 'escape' => '\\'));
        $config = new FormConfig('upload', $subject, $translator);

        $config->addBaseStructure('', '');
        $config->addTab('content/main', 'main');
        $config->addPanel(
            'content/main/help',
            null,
            false,
            array(
                'description' => $translator->trans(
                    'The file to import must be in CSV format and should only have the first sheet; the first line must have the column headers;
the content should be formatted to text and must not have empty rows, formulae or cels with format such colors and borders.'
                )
            )
        );

        $config->addPanel('content/main/import');

        // Cambio la label del pulsante di salvataggio
        $submit_field = $config->getField('actions/submit');
        $submit_field['label'] = $translator->trans('Start import');

        $config->addFields(
            'content/main/import',
            array(
                'file' => array(
                    'class' => 'Pongho\\Form\\Field\\FileField',
                    'label' => $translator->trans('Import file'),
                    'attributes' => array(
                        'accept' => '.csv',
                        'required' => 'required',
                    ),
                    'settings' => array(
                        'filename'    => $this->getImportHelper()->getFilename(),
                        'upload_path' => $this->getImportHelper()->getUploadPath(),
                    ),
                ),
                'delimiter' => array(
                    'class' => 'Pongho\\Form\\Field\\TextField',
                    'label' => $translator->trans('CSV delimiter'),
                    'attributes' => array(
                        'class' => 'smallest',
                    ),
                ),
                'enclosure' => array(
                    'class' => 'Pongho\\Form\\Field\\TextField',
                    'label' => $translator->trans('Text enclosure'),
                    'attributes' => array(
                        'class' => 'smallest',
                    ),
                ),
                'escape' => array(
                    'class' => 'Pongho\\Form\\Field\\TextField',
                    'label' => $translator->trans('Escape char'),
                    'attributes' => array(
                        'class' => 'smallest',
                    ),
                ),
            )
        );

        return $config;
    }

    /**
     * @return FormConfig
     */
    protected function getImportFormConfig()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');
        $subject = new ArrayRowSubject(array());
        $config = new FormConfig('import', $subject, $translator);

        $config->addBaseStructure('', '');
        $config->addTab('content/main', 'main');
        $config->addPanel(
            'content/main/help',
            null,
            false,
            array(
                'description' => $translator->trans(
                    'Now you must select the corresponding type for the columns if the system was not able to do it alone, every column without a header set will be ignored.<br>
If you need to assign a group to the users, you must specify which columns have the mark for the assignation. If the group name does not exist, it will be created as new.'
                )
            )
        );

        $config->addPanel('content/main/import');
//        $import_panel = $config->getField('content/main/import');
//        $import_panel['attributes'] = array('style' => 'overflow: auto;');

        // Cambio la label del pulsante di salvataggio
        $submit_field = $config->getField('actions/submit');
        $submit_field['label'] = $translator->trans('Continue import');

        // Esempio di dati caricati dal file Excel
        $headers = $this->getImportHelper()->getImportHeaders();

        // Configurazione delle colonne
        $columns = array();
        $association = array();

        foreach ($headers as $colname) {
            // Tento di associare le intestazioni delle colonne alle proprietà del modello
            $search = array_search($colname, $this->getColumnsOptions());

            if ($search) {
                $association[$colname] = $search;
            }

            // Configurazione dell'header della colonna
            $columns[] = array(
                'label' => $translator->get($colname),
                'field' => $colname,
                'class' => 'small',
                'callback' => function(TableRow $row) use ($colname, $headers) {
                    // La prima riga gestisce le associazioni
                    if ($row->getKey() === 0) {
                        /** @var \Pongho\Form\Field\SelectField $field */
                        $field = $row->getField($colname);
                        return $field->getTag();
                    }

                    return html_escape($row->getValue($colname));
                },
            );
        }

        // Imposto i dati nel subject
        $subject->set('table',array_merge(
                array( // Riga di associazione
                    new ArrayRowSubject($association),
                ),
                $this->getImportHelper()->getPreviewData()
            )
        );

        $config->addField(
            'content/main/import/table',
            array(
                'class' => 'Pongho\\Form\\Table\\TableFieldset',
                'settings' => array(
                    'columns' => $columns,
                ),
            )
        );

        foreach ($headers as $colname) {
            $config->addField(
                'content/main/import/table/' . $colname,
                array(
                    'class' => 'Pongho\\Form\\Field\\SelectField',
                    'label' => $colname,
                    'attributes' => array(
                        'class' => 'small',
                    ),
                    'options' => $this->getColumnsOptions(),
                )
            );
        }

        return $config;
    }

    /**
     * @return array
     */
    protected function getColumnsOptions()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        return array(
            null             => '',
            'name'           => $translator->trans('Name'),
            'surname'        => $translator->trans('Surname'),
            'email'          => $translator->trans('E-mail'),
            'username'       => $translator->trans('User name'),
            'password'       => $translator->trans('Password'),
            'company'        => $translator->trans('Company'),
            'partita_iva'    => $translator->trans('VAT code'),
            'codice_fiscale' => $translator->trans('Tax code'),
            'address'        => $translator->trans('Address'),
            'city'           => $translator->trans('City'),
            'postcode'       => $translator->trans('Post code'),
            'province'       => $translator->trans('Province'),
            'country'        => $translator->trans('Country'),
            'telephone'      => $translator->trans('Telephone'),
            'fax'            => $translator->trans('Fax'),
            'website'        => $translator->trans('Website'),
            'group'          => $translator->trans('Group'),
        );
    }
}
