<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Entity\AccountInterface;
use Application\Core\Model\AccountPortfolio;
use Application\Core\Model\Permit;
use Application\Core\Model\PermitRole;
use Application\Core\Model\Role;
use Application\Core\Model\Province;
use Application\Core\Model\GroupUser;
use Application\Core\Model\Group;
use Application\Core\Model\File;
use Application\Core\Model\Account;
use Application\Core\Model\Language;
use Monolog\Logger;
use Pongho\Form\Field;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Filter\EmailFilter;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;
use Pongho\Utilities\Inflector;

/**
 * Class UsersController
 */
class UsersController extends CrudFormController
{
    private $language_count;

    /**
     * Variabile usata per conservare la password in chiaro in caso di notifica all'utente
     *
     * @var string
     */
    protected $password;

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return 'Application\\Core\\Model\\Account';
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($this->getAction() === 'add') {
            return $translator->trans('Add a new user');
        } else {
            return $translator->trans('Edit the user %name%', array('%name%' => $model->name()));
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'users';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.users.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getArchiveTitle()
    {
        return $this->getContainer()->getService('translator')->trans('Users list');
    }

    /**
     * {@inheritdoc}
     */
    public function getArchiveAddButtonText()
    {
        return $this->getContainer()->getService('translator')->trans('Add a new user');
    }

    /**
     * {@inheritdoc}
     */
    public function hasAddButton()
    {
        return $this->getHelper()->userHasPermit('users.add');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getContainer()->getService('translator')->trans('Add the first user');
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        if ($this->getHelper()->userHasPermit('users.delete')) {
            return parent::getMassActions();
        }

        return array();
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        if ($this->getHelper()->getUser()->isFounder()) {
            $model_options = array(
                'select'     => '`from`.*',
                'conditions' => array(
                    '`from`.id <> :anonymous',
                    'anonymous' => Account::ANONYMOUS,
                )
            );
        } elseif (!Permit::isEnabled('users.view_all_users') || $this->getHelper()->userHasPermit('users.view_all_users')) {
            $model_options = array(
                'select'     => '`from`.*',
                'conditions' => array(
                    '`from`.id <> :anonymous AND `from`.is_founder <> :is_founder',
                    'anonymous'  => Account::ANONYMOUS,
                    'is_founder' => true,
                )
            );
        } else {
            $model_options = array(
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . AccountPortfolio::tableName() . ' AS p ON p.descendant_id = `from`.id',
                'conditions' => array(
                    '`from`.id <> :anonymous AND `from`.is_founder <> :is_founder AND p.ancestor_id = :ancestor',
                    'anonymous'  => Account::ANONYMOUS,
                    'is_founder' => true,
                    'ancestor'   => $this->getHelper()->getUserId(),
                )
            );
        }

        $model_options['from'] = Account::tableName() . ' AS `from`';

        /** @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = $this->getContainer()->getService('application_manager');

        if ($app_manager->isInstalled('Showcase')) {
            $model_options['include'] = array('total_orders');
        }

        // Ordinamento per data di creazione, dal più recente al più vecchio
        $model_options['order'] = 'created_at DESC';

        return $model_options;
    }

    /**
     * {@inheritdoc}
     */
    protected function getSearchFields()
    {
        return array('name', 'surname', 'fullname', 'email', 'company');
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        return array(
            array(
                'name'      => 'fullname',
                'label'     => $translator->trans('Name'),
                'class'     => 'main',
                'orderable' => true,
                'render'    => function (Account $row) {
                        return '<strong>' . $row->name() . '</strong><div class="actions"><span>' . $row->actions . '</span></div>';
                    },
            ),
            array(
                'name'      => 'company',
                'label'     => $translator->trans('Company'),
                'class'     => 'large',
                'orderable' => true,
            ),
            array(
                'name'      => 'username',
                'label'     => $translator->trans('Username'),
                'class'     => 'large',
                'orderable' => true,
            ),
            array(
                'name'      => 'email',
                'label'     => $translator->trans('E-Mail address'),
                'class'     => 'large',
                'orderable' => true,
            ),
            array(
                'name'      => 'created_at',
                'label'     => $translator->trans('Subscribe date'),
                'orderable' => true,
                'render'    => function (Account $row) {
                        return $row->created_at->format('d-m-Y');
                    }
            ),
            array(
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ),
        );
    }

    /**
     * {@inheritdoc}
     */
    public function parseArchiveRow($row)
    {
        /** @var Account $row */

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $actions = array();

        $path = $this->getHelper()->getPath();

        if ($this->getHelper()->userHasPermit("users.edit")) {
            $actions[] = array($translator->trans('Edit'), 'href' => $this->url("/{$path}edit/{$row->id}/"), 'class' => 'edit');
        }

        if ($this->getHelper()->userHasPermit('users.impersonation') && $row->id != $this->getHelper()->getUserId()) {
            $actions[] = array($translator->trans('Impersonate'), 'href' => url('/?switch_user=' . $row->id));
        }

        if ($this->getHelper()->userHasPermit("users.delete") && $row->id != $this->getHelper()->getUserId() && $row->isDeletable()) {
            $actions[] = array($translator->trans('Delete'), 'href' => $this->url("/{$path}delete/{$row->id}/"), 'class' => 'delete');
        }

        $row->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelFieldsConfig()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        return array(
            'name'           => array('label' => $translator->trans('Name')),
            'surname'        => array('label' => $translator->trans('Surname')),
            'username'       => array('label' => $translator->trans('Username')),
            'email'          => array('label' => $translator->trans('Email')),
            'created_at'     => array('label' => $translator->trans('Subscribe date')),
            'language_id'    => array('label' => $translator->trans('Language'), 'data-type' => 'foreign'),
            'role_id'        => array('label' => $translator->trans('Role'), 'data-type' => 'foreign'),
            'group'          => array(
                'label'     => $translator->trans('Group'),
                'data-type' => 'foreign',
                'column'    => 'g.group_id',
                'join'      => 'LEFT JOIN ' . GroupUser::tableName() . ' AS g ON g.user_id = `from`.id'
            ),
            'address'        => array('label' => $translator->trans('Address')),
            'address2'       => array('label' => $translator->trans('Address info')),
            'city'           => array('label' => $translator->trans('City')),
            'postcode'       => array('label' => $translator->trans('Post code')),
            'province_id'    => array('label' => $translator->trans('Province'), 'data-type' => 'foreign'),
            'company'        => array('label' => $translator->trans('Company')),
            'fax'            => array('label' => $translator->trans('Fax')),
            'telephone'      => array('label' => $translator->trans('Telephone')),
            'partita_iva'    => array('label' => $translator->trans('VAT code')),
            'codice_fiscale' => array('label' => $translator->trans('Tax code')),
            'is_active'      => array('label' => $translator->trans('User status')),
            'newsletter'     => array('label' => $translator->trans('Newsletter')),
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getFilterEnumValues()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $boolean = array(
            '0' => $translator->trans('Not active'),
            '1' => $translator->trans('Active'),
        );

        return array(
            'is_active'  => array_merge(
                array(null => $translator->trans('Pending activation')),
                $boolean
            ),
            'newsletter' => $boolean,
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getFilterSelectValues()
    {
        /** @var \Pongho\Template\Html $html */
        $html = $this->getContainer()->getService('template_html');

        return array(
            'language_id' => function () {
                    $options = array(
                        'select' => 'id, name',
                        'order'  => 'name ASC',
                    );

                    $o = array();

                    /** @var Language $row */
                    foreach (Language::all($options) as $row) {
                        $o[$row->id] = $row->name;
                    }

                    return $o;
                },
            'province_id' => function () use ($html) {
                    $options = array(
                        'select' => 'id, name, plate',
                        'order'  => 'name ASC',
                    );

                    $o = array();

                    /** @var Province $row */
                    foreach (Province::all($options) as $row) {
                        $o[$row->id] = $row->name;
                    }

                    return $html->select($o);
                },
            'role_id'     => function () {
                    $options = array(
                        'select'     => 'id, name',
                        'conditions' => array('id <> ?', Account::ANONYMOUS),
                        'order'      => 'name ASC',
                    );

                    $o = array();

                    /** @var Role $row */
                    foreach (Role::all($options) as $row) {
                        $o[$row->id] = $row->name;
                    }

                    return $o;
                },
            'group'       => function () {
                    $options = array(
                        'select' => 'id, name',
                        'order'  => 'name ASC',
                    );

                    $o = array();

                    /** @var Group $row */
                    foreach (Group::all($options) as $row) {
                        $o[$row->id] = $row->name;
                    }

                    return $o;
                },
        );
    }

    /**
     * @return int
     */
    private function getLanguageCount()
    {
        if ($this->language_count === null) {
            $this->language_count = Language::count();
        }

        return $this->language_count;
    }

    /**
     * {@inheritdoc}
     */
    protected function isSubmit()
    {
        return ($this->getRequest()->getMethod() === 'POST') || $this->isNotify() || $this->isCredentialsNotify();
    }

    /**
     * Restituisce true se devo inviare la notifica al cliente. Altrimenti false.
     *
     * @return boolean
     */
    protected function isNotify()
    {
        $form_request = $this->getRequest()->post->get('user', array());

        return (isset($form_request['notify_edit']) && $form_request['notify_edit'] === 'on');
    }

    /**
     * Restituisce true se devo comunicare delle nuove credenziali di accesso all'utente
     *
     * @return bool
     */
    protected function isCredentialsNotify()
    {
        $form_request = $this->getRequest()->post->get('user', array());

        return (isset($form_request['notify_credentials']) && $form_request['notify_credentials'] === 'on');
    }

    /**
     * Restituisce true se devo comunicare un nuovo link di attivazione all'utente
     *
     * @return bool
     */
    protected function isActivationLinkNotify()
    {
        $form_request = $this->getRequest()->post->get('user', array());

        return (isset($form_request['notify_activation_link']) && $form_request['notify_activation_link'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isCreationNotify()
    {
        $form_request = $this->getRequest()->post->get('user', array());

        return (isset($form_request['notify_create']) && $form_request['notify_create'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isStatusChangeNotify()
    {
        $form_request = $this->getRequest()->post->get('user', array());

        return (isset($form_request['notify_status']) && $form_request['notify_status'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isActivationNotify()
    {
        return ($this->isStatusChangeNotify() && $this->getModel()->isActive() === true);
    }

    /**
     * @return bool
     */
    protected function isDeactivationNotify()
    {
        return ($this->isStatusChangeNotify() && $this->getModel()->isActive() === false);
    }

    /**
     * Callback before_update
     *
     * @param Account $account
     * @param array   $fields
     */
    protected function beforeUpdateModelCallback(Account $account, array $fields)
    {
        $this->getHelper()->notify(
            $this,
            $this->getEventNamespace('before_update_model'),
            array('account' => $account, 'fields' => $fields)
        );
    }

    /**
     * Callback before_save
     *
     * @param Form $form
     */
    public function beforeSaveCallback(Form $form)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $form->getSubject()->getModel();

        // Serve per la validazione del modello
        $account->privacy = true;

        // Faccio in modo di non poter disattivare se stessi
        if ($account->id == $this->getHelper()->getUserId()) {
            $account->activate();
        }

        $this->getHelper()->notify($this, $this->getEventNamespace('before_save'), array('account' => $account));
    }

    /**
     * Callback after_save
     *
     * @param Form $form
     */
    protected function afterSaveCallback(Form $form)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $form->getSubject()->getModel();

        // Invio la notifica se ho selezionato qualche spunta.
        $this->sendNotifications($account);

        $this->getHelper()->notify($this, $this->getEventNamespace('after_save'), array('account' => $account));
    }

    /**
     * @param ModelSubject $subject
     * @param              $value
     */
    public function setPasswordCallback(ModelSubject $subject, $value)
    {
        // Se sto inviando una notifica all'utente
        if (empty($value) && ($this->isCredentialsNotify() || $this->isCreationNotify())) {
            $value = random(8);
        }

        // Backup della password in chiaro da usare per le notifiche
        $this->password = $value;

        /** @var \Application\Core\Model\Account $account */
        $account = $subject->getModel();

        $account->password = $value;
        $account->confirm_password = $value;
    }

    /**
     * @param ModelSubject $subject
     * @param              $value
     * @throws \UnexpectedValueException
     */
    public function setAvatarCallback(ModelSubject $subject, $value)
    {
        if (!is_array($value)) {
            throw new \UnexpectedValueException('The value returned for the "avatar" field must be an array');
        }

        /** @var \Application\Core\Model\Site $site */
        $site = $this->getContainer()->getService('main_site');

        $value['site_id'] = $site->id;
        $value['path'] = substr($value['path'], strlen($site->getUploadsPath()));

        /** @var Account $account */
        $account =  $subject->getModel();

        if (is_numeric($file_id = $account->avatar_id)) {
            $file = File::find($file_id);

            $account->avatar_id = null;
            $account->save(false);

            $file->delete();
        }

        /** @var File $file */
        $file = File::create($value);

        $account->avatar_id = $file->id;
    }

    /**
     * @param ModelSubject $subject
     * @return array
     */
    public function getGroupsCallback(ModelSubject $subject)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        return array_keys($model->getGroups());
    }

    /**
     * @param ModelSubject $subject
     * @param              $value
     */
    public function setGroupsCallback(ModelSubject $subject, $value)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        $model->setGroups($value);
    }

    /**
     * Callback per il subject della form.
     *
     * Restituisce l’elenco degli ID dei genitori diretti (primo livello) dell’utente.
     *
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @return array
     */
    public function getSubjectParentsCallback(ModelSubject $subject)
    {
        /** @var \Application\Core\Model\Account $model */
        $model = $subject->getModel();

        $parents = array();
        foreach ($model->getParents() as $parent) {
            $parents[] = $parent->id;
        }

        return $parents;
    }

    /**
     * @param ModelSubject $subject
     * @param              $value
     */
    public function setSubjectParentsCallback(ModelSubject $subject, $value)
    {
        /** @var \Application\Core\Model\Account $model */
        $model = $subject->getModel();

        $model->setParents($value);
    }

    /**
     * @param bool|null $status
     * @return string
     */
    protected function accountIsActiveToString($status)
    {
        if ($status === null) { return 'pending'; }
        if ($status === false) { return 'not active'; }
        if ($status === true) { return 'active'; }

        throw new \InvalidArgumentException('The value "' . $status . '" is not valid');
    }

    /**
     * @param string $status
     * @return bool|null
     */
    protected function accountIsActiveFromString($status)
    {
        if ($status === 'pending') { return null; }
        if ($status === 'not active') { return false; }
        if ($status === 'active') { return true; }

        throw new \InvalidArgumentException('The value "' . $status . '" is not valid');
    }

    /**
     * @param ModelSubject $subject
     * @return string
     */
    public function getSubjectIsActiveCallback($subject)
    {
        /** @var AccountInterface $model */
        $model = $subject->getModel();

        return $this->accountIsActiveToString($model->isActive());
    }

    /**
     * @param ModelSubject $subject
     * @param              $value
     */
    public function setSubjectIsActiveCallback(ModelSubject $subject, $value)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        $value = $this->accountIsActiveFromString($value);

        if (!$model->isActive() && $value) {
            $this->getHelper()->notify($this, 'account_activation', array('account' => $subject->getModel()));
        }

        $model->is_active = $value;
    }

    /**
     * @var Account
     */
    protected $account;

    /**
     * {@inheritdoc}
     */
    protected function getModel()
    {
        if ($this->account === null) {
            $this->account = parent::getModel();

            if ($this->getAction() == 'add') {
                $site = $this->getHelper()->getSite();

                $this->account->language_id = $site->default_language_id;
                $this->account->password = random(8);
                $this->account->is_active = $site->getOption('subscription_mode') === 'simple' ? true : null;
                $this->account->newsletter = true;
            }
        }

        return $this->account;
    }

    /**
     * Restituisce il Subject per la Form
     *
     * @return \Pongho\Form\Subject\ModelSubject
     */
    protected function getSubject()
    {
        $class = $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject_class'), 'Pongho\\Form\\Subject\\ModelSubject');

        /** @var \Pongho\Form\Subject\ModelSubject $subject */
        $subject = new $class($this->getModel());

        // Collego la gestione dei tipi di campo particolari al subject
        $subject
            ->attachSetter('password', array($this, 'setPasswordCallback'))
            ->attachSetter('avatar_id', array($this, 'setAvatarCallback'))

            ->attachGetter('groups', array($this, 'getGroupsCallback'))
            ->attachSetter('groups', array($this, 'setGroupsCallback'))

            ->attachGetter('ancestors', array($this, 'getSubjectParentsCallback'))
            ->attachSetter('ancestors', array($this, 'setSubjectParentsCallback'))

            ->attachGetter('is_active', array($this, 'getSubjectIsActiveCallback'))
            ->attachSetter('is_active', array($this, 'setSubjectIsActiveCallback'));

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * {@inheritdoc}
     */
    public function getFormsOptions()
    {
        $parameters = array('subject' => $this->getSubject());

        $options = $this->getDefaultFormsOptions();
        $options = $this->getHelper()->filter($this, $this->getEventNamespace('filter_form_options'), $options, $parameters);

        return $options;
    }

    /**
     * Restituisce le opzioni di base della form.
     *
     * @access protected
     * @return \Pongho\Form\Utilities\FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        /** @var \Pongho\Form\Subject\ModelSubject $subject */
        $subject = $this->getSubject();

        /** @var \Application\Core\Model\Account $account */
        $account = $subject->getModel();

        /** @var \Application\Core\Model\Site $site */
        $site = $this->getContainer()->getService('main_site');

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        // Gestione del campo password
        $password_field = array(
            'class'       => 'Pongho\\Form\\Field\\PasswordField',
            'label'       => $translator->trans('Password'),
            'attributes'  => array(
                'class' => 'input_text',
                'autocomplete' => 'off',
                'placeholder' => $translator->trans('Fill out only for change'),
            ),
        );

        if ($this->action === 'add') {
            $password_field['attributes']['required'] = 'required';
            unset($password_field['attributes']['placeholder']);
        }

        // Gestione del campo language
        if ($this->getLanguageCount() > 1) {
            $language_field = array(
                'class' => 'Application\\Core\\Form\\Field\\LanguageField',
                'label' => $translator->trans('Language'),
            );
        } else {
            $language_field = array('class' => 'Pongho\\Form\\Field\\HiddenField');
        }

        $config = new FormConfig('user', $subject, $this->getHelper()->getLocalization());

        $config->addBaseStructure($this->getActiveTab(), $this->currentUrl());


        // Pannello notifiche
        $config->addPanelBefore(
            'actions',
            'notifications',
            'Notifiche',
            false,
            array(
                'description' => $translator->trans(
                    'Will be sent an email for each selected notification. The notifications can be sent only to active
                     accounts or in the activation phase, with the exception of the notification of deactivation. Some
                     notifications can be selected only after certain operations.'
                ),
            )
        );

        if ($this->getAction() === 'edit') {
            $attributes = array();

            if (!$account->isActive()) {
                $attributes = array(
                    'disabled' => 'disabled',
                );
            }

            if ($account->isActive() === null) {
                $config->addField(
                    'notifications/notify_activation_link',
                    array(
                        'class'      => 'Pongho\\Form\\Field\\CheckboxField',
                        'label'      => $translator->trans('Activation link'),
                        'description' => $translator->trans(
                            'Will be sent a new single use activation link'
                        ),
                    )
                );
            }

            $config->addField(
                'notifications/notify_edit',
                array(
                    'class'      => 'Pongho\\Form\\Field\\CheckboxField',
                    'label'      => $translator->trans('Account edit'),
                    'description' => $translator->trans(
                        'Will not list the effective changes, it only notify the user about the change'
                    ),
                    'attributes' => $attributes,
                )
            );

            $config->addField(
                'notifications/notify_status',
                array(
                    'class' => 'Pongho\\Form\\Field\\CheckboxField',
                    'label' => $translator->trans('Status change'),
                    'description' => $translator->trans(
                        'Will be sent a notify about the status change'
                    ),
                    'attributes' => array(
                        'disabled' => 'disabled',
                    ),
                )
            );

            $config->addField(
                'notifications/notify_credentials',
                array(
                    'class'       => 'Pongho\\Form\\Field\\CheckboxField',
                    'label'       => $translator->trans('New access credentials'),
                    'description' => $translator->trans(
                        'Will be sent the new password entered in the field or a random one if left empty'
                    ),
                    'attributes'  => $attributes,
                )
            );
        } else {
            $config->addField(
                'notifications/notify_create',
                array(
                    'class'      => 'Pongho\\Form\\Field\\CheckboxField',
                    'label'      => $translator->trans('Account creation'),
                    'description' => $translator->trans(
                        'Will be sent the credentials, and the activation link if applicable, for the new account'
                    ),
                )
            );
        }

        $config->addField(
            'content/main',
            array(
                'class' => 'Application\\Admin\\Form\\PanelFieldset',
                'label' => $translator->trans('Main'),
            )
        );

        $config->addFields(
            'content/main',
            array(
                'account' => array(
                    'class'      => 'Pongho\\Form\\Fieldset',
                    'attributes' => array('class' => 'panel'),
                    'label'      => $translator->trans('Account'),
                ),
                'profile' => array(
                    'class'      => 'Pongho\\Form\\Fieldset',
                    'attributes' => array('class' => 'panel'),
                    'label'      => $translator->trans('Profile'),
                ),
            )
        );

        $config->addFields(
            'content/main/account',
            array(
                'username'    => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Username'),
                    'attributes' => array('class' => 'input_text', 'required' => 'required', 'autocomplete' => 'off')
                ),
                'password'    => $password_field,
                'email'       => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('E-Mail address'),
                    'attributes' => array('class' => 'input_text', 'required' => 'required', 'autocomplete' => 'off'),
                    'filter'     => new EmailFilter()
                ),
                'language_id' => $language_field,
                'role_id'     => array(
                    'class' => 'Pongho\\Form\\Field\\SelectField',
                    'label' => $translator->trans('Role'),
                    'options' => array($this, 'getRolesOptionsCallback'),
                ),
                'is_active'   => array(
                    'class'   => 'Pongho\\Form\\Field\\SelectField',
                    'label'   => $translator->trans('User status'),
                    'options' => array($this, 'getStatusOptionsCallback'),
                ),
                'newsletter'  => array(
                    'class'       => 'Pongho\\Form\\Field\\CheckboxField',
                    'label'       => $translator->trans('Newsletter'),
                    'description' => $translator->trans('The user can receive the newsletter')
                ),
            )
        );

        $config->addFields(
            'content/main/profile',
            array(
                'avatar_id'   => array(
                    'class'       => 'Pongho\\Form\\Field\\ImageField',
                    'label'       => $translator->trans('Avatar'),
                    'settings'    => array(
                        'delete_url'  => $this->url('/users/deleteavatar/' . $account->id . '/'),
                        'preview_url' => function () use ($account) {
                                if (($avatar = $account->getAvatar()) !== null) {
                                    return $avatar->path();
                                }
                                return '';
                            },
                        'upload_path' => $site->getUploadsPath('/avatars/'),
                        //'allowed_extensions'	=> array(),
                        //'max_file_size'		=> null,
                        //'filename'			=> null,
                    ),
                ),
                'description' => array(
                    'class' => 'Pongho\\Form\\Field\\TextareaField',
                    'label' => $translator->trans('Description')
                ),
            )
        );

        if ($this->getHelper()->getSite()->getOption('account_url') && $this->getContainer()->getService('router')->has('user-view')) {
            $config->addFieldAfter(
                'content/main/account/email',
                'url',
                array(
                    'class'       => 'Application\\Core\\Form\\Field\\AccountUrlField',
                    'label'       => $translator->trans('Account public url'),
                    'description' => $translator->trans('Allows to change the url to access the user\'s public profile'),
                    'settings'    => array(
                        'url-path'       => absolute_url($this->getContainer()->getService('router')->get('user-view')->getPath()) . '/',
                        'button-value'   => $translator->trans('Edit'),
                        'form-url'       => $this->url('/' . $this->getParameter('path') . '/changeurl/' . $account->id . '/'),
                        'lightbox-title' => $translator->trans('Edit the user\'s url'),
                        'disable-edit'   => ($account->old_url !== ''),
                    ),
                )
            );
        }

        $config->addFields(
            'content',
            array(
                'details' => array(
                    'class' => 'Application\\Admin\\Form\\PanelFieldset',
                    'label' => $translator->trans('Details'),
                ),
            )
        );

        $config->addFields(
            'content/details',
            array(
                'user-details' => array(
                    'class'      => 'Pongho\\Form\\Fieldset',
                    'attributes' => array('class' => 'panel'),
                    'label'      => $translator->trans('User data'),
                ),
                'user-address' => array(
                    'class'      => 'Pongho\\Form\\Fieldset',
                    'attributes' => array('class' => 'panel'),
                    'label'      => $translator->trans('Address'),
                ),
            )
        );

        $config->addFields(
            'content/details/user-details',
            array(
                'name'           => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Name'),
                    'attributes' => array('class' => 'input_text')
                ),
                'surname'        => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Surname'),
                    'attributes' => array('class' => 'input_text')
                ),
                'company'        => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Company'),
                    'attributes' => array('class' => 'input_text')
                ),
                'partita_iva'    => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('VAT'),
                    'attributes' => array('class' => 'input_text')
                ),
                'codice_fiscale' => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Tax code'),
                    'attributes' => array('class' => 'input_text')
                ),
                'telephone'      => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Telephone'),
                    'attributes' => array('class' => 'input_text')
                ),
                'fax'            => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Fax'),
                    'attributes' => array('class' => 'input_text')
                ),
                'website'        => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Website'),
                    'attributes' => array('class' => 'input_text')
                ),
                'googleplus'     => array(
                    'class'       => 'Pongho\\Form\\Field\\TextField',
                    'label'       => $translator->trans('Google+ profile'),
                    'attributes'  => array('class' => 'input_text'),
                    'description' => $translator->trans('Enter the URL of your Google+ profile to connect the articles you have written on this site.'),
                ),
            )
        );

        $config->addFields(
            'content/details/user-address',
            array(
                'address'     => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Address'),
                    'attributes' => array('class' => 'input_text')
                ),
                'address2'    => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Address info'),
                    'attributes' => array('class' => 'input_text')
                ),
                'city'        => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('City'),
                    'attributes' => array('class' => 'input_text')
                ),
                'postcode'    => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'label'      => $translator->trans('Post code'),
                    'attributes' => array('class' => 'input_text')
                ),
                'province_id' => array(
                    'class'    => 'Application\\Core\\Form\\Field\\ProvinceField',
                    'label'    => $translator->trans('Province'),
                    'settings' => array(
                        'placeholder' => $translator->trans('Select a province'),
                    ),
                ),
            )
        );

        // Visualizzo le altre form solo se l'azione non è add perché non posso salvare un nuovo utente dalla tab dei gruppi.
        if ($this->action !== 'add') {
            if ($this->getHelper()->userHasPermit('users.handle_groups')) {
                $config->addTab('content/groups', $translator->trans('Groups'));
                $config->addPanel('content/groups/groups-panel');

                $groups = array();
                /** @var Group $group */
                foreach (Group::all() as $group) {
                    $groups[$group->id] = '<strong>' . $group->name . '</strong><br><em>' . $group->description . '</em>';
                }

                $config->addField(
                    'content/groups/groups-panel/groups',
                    array(
                        'class'    => 'Application\\Admin\\Form\\Field\\TableChecklistField',
                        'options'  => $groups,
                        'settings' => array(
                            'header_label' => $translator->trans('Group'),
                        ),
                    )
                );
            }

            if ($this->getHelper()->userHasPermit('users.handle_assoc') && $this->getHelper()->getUserId() != $account->id) {
                $config->addTab('content/ancestors', $translator->trans('Ancestors'));
                $config->addPanel('content/ancestors/ancestors-panel');

                $options['ancestors'] = new FormConfig('details', $subject, $this->getHelper()->getLocalization());

                $ancestors_options = array();

                $joins = array();
                $conditions = array(
                    '`from`.id NOT IN(:anonymous, :self)',
                    'anonymous' => Account::ANONYMOUS,
                    'self'      => $account->id
                );

                $joins[] = 'LEFT JOIN ' . AccountPortfolio::tableName() . ' AS descendant ON descendant.descendant_id = `from`.id';

                // Gli utenti presenti in questo pannello devono essere discendenti dell’utente corrente
                if (!$this->getHelper()->userHasPermit('users.view_all_users')) {
                    $conditions[0] .= ' AND descendant.ancestor_id = :current_user';
                    $conditions['current_user'] = $this->getHelper()->getUserId();
                }

                // Gli utenti presenti in questo pannello NON devono essere discendenti dell’utente che si sta modificando
                $conditions[0] .= ' AND descendant.ancestor_id <> :account';
                $conditions['account'] = $this->getHelper()->getUserId();

                // Gli utenti presenti in questo pannello devono avere il permesso 'is_ancestor'
                $joins[] = 'LEFT JOIN ' . PermitRole::tableName() . ' AS pr ON pr.role_id = `from`.role_id';
                $joins[] = 'LEFT JOIN ' . Permit::tableName() . ' AS p ON p.id = pr.permit_id';

                $conditions[0] .= ' AND p.`key` = :permit_key';
                $conditions['permit_key'] = 'users.is_ancestor';

                // Compongo la query
                if (!empty($joins)) {
                    $ancestors_options['select'] = '`from`.*';
                    $ancestors_options['joins'] = implode(' ', $joins);
                }

                $ancestors_options['conditions'] = $conditions;

                /** @var \Application\Core\Model\Account $ancestor */
                $ancestors = array();
                foreach (Account::all($ancestors_options) as $ancestor) {
                    $ancestors[$ancestor->id] = '<strong>' . $ancestor->name() . '</strong>';
                }

                $config->addField(
                    'content/ancestors/ancestors-panel/ancestors',
                    array(
                        'class'    => 'Application\\Admin\\Form\\Field\\TableChecklistField',
                        'options'  => $ancestors,
                        'settings' => array(
                            'header_label' => $translator->trans('Ancestor'),
                        ),
                    )
                );
            }
        }

        return $config;
    }

    /**
     * @return array
     *
     * @todo RoleManager
     */
    public function getRolesOptionsCallback()
    {
        $query_options = array('conditions' => array('id <> ?', Role::USER_NOT_LOGGED));
        $select_options = array();

        /** @var Role $role */
        foreach (Role::all($query_options) as $role) {
            $select_options[$role->id] = $role->name;
        }

        return $select_options;
    }

    /**
     * @param SelectField $field
     * @return array
     */
    public function getStatusOptionsCallback(SelectField $field)
    {
        /** @var Account $account */
        $account = $field->getForm()->getSubject()->getModel();

        $options = array();

        if ($account->isActive() === null) {
            $options['pending'] = $this->getHelper()->getTranslator()->trans('Pending activation');
        }

        $options['not active'] = $this->getHelper()->getTranslator()->trans('Not active');
        $options['active'] = $this->getHelper()->getTranslator()->trans('Active');

        return $options;
    }

    /**
     * @param Account $account
     */
    protected function sendNotifications(Account $account)
    {
        $this->prepareNotifications($account);

        $this->getNotificationHelper()->send();
    }

    /**
     * @param Account $account
     */
    protected function prepareNotifications(Account $account)
    {
        /** @var Logger $logger */
        $logger = $this->getContainer()->getService('logger');

        $notification_helper = $this->getNotificationHelper();
        $admin_account = $this->getHelper()->getUser()->getAccount();

        $data = array(
            'user' => array(
                'id'     => $account->getId(),
                'status' => $this->accountIsActiveToString($account->isActive()),
            ),
            'author' => array(
                'id'       => $admin_account->getId(),
                'username' => $admin_account->getUsername(),
            ),
        );

        if ($this->isNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Account edit notify', $data);

            if (!$notification_helper->addUserEditNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore account edit notify', $data);
            }
        }

        if ($this->isStatusChangeNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Status change notify', $data);

            if (!$notification_helper->addUserStatusChangeNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore status change notify', $data);
            }
        }

        if ($this->isCredentialsNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Send credentials notify', $data);

            if (!$notification_helper->addUserCredentialsNotification($account, $this->password)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore send credentials notify', $data);
            }
        }

        if ($this->isActivationLinkNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Send activation link notify', $data);

            if (!$notification_helper->addUserActivationLinkNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore send activation link notify', $data);
            }
        }

        if ($this->isCreationNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Account create notify', $data);

            if (!$notification_helper->addUserCreationNotification($account, $this->password)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore account create notify', $data);
            }
        }
    }

    /**
     * Restituisce l’aiutante per le modifiche.
     *
     * Sovrascrivendo questo metodo è possibile sostituire l’aiutante con uno scritto ad hoc.
     *
     * @return \Application\Core\Utilities\AdminUserNotificationHelper
     */
    protected function getNotificationHelper()
    {
        return $this->getContainer()->getService('admin_user_notification_helper');
    }

    /**
     * @param AccountInterface $account
     * @return string
     *
     * @deprecated Usare il servizio "user_activation_link_builder"
     */
    protected function getUserActivationLink(AccountInterface $account)
    {
        /** @var \Application\Core\Utilities\UserActivationLinkBuilder $builder */
        $builder = $this->getContainer()->getService('user_activation_link_builder');

        return $builder->build($account);
    }

    /**
     * @return JsonResponse|RedirectResponse
     */
    public function deleteavatarAction()
    {
        $id = $this->getParameter('id', null, true);

        /** @var Account $account */
        $account = Account::find($id);
        $account->deleteAvatar();

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayJsonMessage($this->getContainer()->getService('translator')->trans('Avatar deleted'));
        }

        return new RedirectResponse($this->url(sprintf('/%s/edit/%d/', $this->getParameter('path'), $id)));
    }

    /**
     * {@inheritdoc}
     */
    protected function checkAddPermit()
    {
        if (!$this->getHelper()->userHasPermit('users.add')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function checkEditPermit()
    {
        $this->checkUserEditPermit();
        $this->checkPortfolioRelationPermit();
    }

    /**
     * {@inheritdoc}
     */
    protected function checkDeletePermit()
    {
        $this->checkUserDeletePermit();
        $this->checkPortfolioRelationPermit();
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkUserEditPermit()
    {
        if (!$this->getHelper()->userHasPermit('users.edit')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkUserDeletePermit()
    {
        if (!$this->getHelper()->userHasPermit('users.delete')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkPortfolioRelationPermit()
    {
        if ($this->getHelper()->userHasPermit('users.is_ancestor') && !$this->getHelper()->userHasPermit('users.view_all_users')) {
            $account = $this->getModel();

            if (!$this->getHelper()->getUser()->getAccount()->hasDescendant($account->id)) {

                // Eseguo un controllo di coerenza, nel caso l'utente corrente non riesca a vedersi perché manca la relazione
                if ($this->getHelper()->getUserId() === $account->id) {
                    throw new \LogicException(sprintf(
                        'The relation for the user %s with himself is missing, check the database',
                        $this->getHelper()->getUserId()
                    ));
                }

                throw new HttpUnauthorizedException();
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function renderAddEdit(Form $form)
    {
        parent::renderAddEdit($form);

        $this->getHelper()->getHead()
            ->addCss(pongho_url('/Application/Core/Resources/views/css/style.css?v=' . filemtime(__DIR__ . '/../../Resources/views/css/style.css')))
            ->addJavaScript(pongho_url('/Application/Core/Resources/views/js/plugins.js?v=' . filemtime(__DIR__ . '/../../Resources/views/js/plugins.js')))
            ->addJavaScript(pongho_url('/Application/Core/Resources/views/js/scripts.js?v=' . filemtime(__DIR__ . '/../../Resources/views/js/scripts.js')));

        $this->getHelper()->getHeaderView()
            ->assignVar('body_id', 'new-layout');

        return null;
    }

    /**
     * Controlla la validità dell'url che si vuole impostare per l'account
     *
     * @param $url
     * @return \Pongho\Http\JsonResponse|string
     */
    protected function checkUrl($url)
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        // Controllo che non inizi con un numero (serebbe meglio un'espressione regolare)
        if (is_numeric(substr($url, 0, 1))) {
            return $this->getHelper()->displayJsonError($translator->trans('The account url must not start with a number'));
        }

        $url = Inflector::urlize($url);

        // Controllo se l'url è già stato usato da altri utenti
        if (Account::find(array('conditions' => array('url = :url OR old_url = :url', 'url' => $url)))) {
            return $this->getHelper()->displayJsonError($translator->trans('The account url is already used by another user'));
        }

        return $url;
    }

    /**
     * Azione per cambiare l'url
     *
     * @return \Pongho\Http\JsonResponse
     */
    public function changeurlAction()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($this->getRequest()->getMethod() === 'POST') {
            $url = trim($this->getRequest()->post->get('url'));

            // Controllo che mi sia stato passato un url e non un testo vuoto
            if (!$url) {
                return $this->getHelper()->displayJsonError($translator->trans('The account url is not defined'));
            }

            if (($url = $this->checkUrl($url)) instanceof JsonResponse) {
                return $url;
            }

            $message = $translator->trans('The url has been verified, confirm to save');

            if ($this->getRequest()->post->get('action') === 'confirm') {
                /** @var Account $account */
                $account = Account::find($this->getParameter('id'));
                $account->old_url = $account->url;
                $account->url = $url;
                $account->save();

                $message = $translator->trans('The url has been modified');
            }

            return $this->getHelper()->displayJsonMessage($message, false, array('url' => $url));
        }

        /** @var Account $account */
        $account = Account::find($this->getParameter('id'));
        $form_action = $this->url('/' . $this->getParameter('path') . '/changeurl/' . $account->id . '/');

        /*
         * # The translation contain HTML tags
         *   the \n at the end of the lines are used for better legibility and are not part of the translation itself
         */
        $description = $translator->trans('<p>Edit the url to access the user\'s public profile.<br>
The url can be modified only once, chose it with care.<br>
The url must be unique, if another user has already chosen the url you want, you must chose a different one.<br>
Only lowercase letters, numbers and the hypen are allowed, the url must not start with a number.<br>
The `Verify` function will check the url validity and will remove any not allowed character.</p>');
        $url = $translator->trans('Url');
        $verify = $translator->trans('Verify');
        $confirm = $translator->trans('Confirm');

        $html = <<<HTML
    <div id="account-url-description">{$description}</div>
    <form id="account-url-form" action="{$form_action}" method="post">
        <div id="account-url-errors"></div>
        <div class="control-group">
            <label class="control-label" for="account-url">{$url}*</label>
            <div class="controls">
                <input id="account-url" required name="url" class="input_text" value="{$account->url}">
            </div>
        </div>
        <button id="account-url-preview" name="preview">{$verify}</button>
        <button id="account-url-confirm" name="confirm" style="display:none;">{$confirm}</button>
    </form>
HTML;

        return new Response($html);
    }
}
