<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Model\LanguageModule;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\Module;
use Application\Core\Model\ModuleSite;
use Application\Core\Model\Translation;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * SitesController
 */
class SitesController extends CrudFormController
{
    /**
     * language_site_id della traduzione da copiare quando si crea un nuovo sito.
     *
     * @access private
     * @var int
     */
    private $source_language_site_id;

    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        if (!$this->getHelper()->getUser()->isFounder()) {
            throw new HttpUnauthorizedException();
        }

        parent::boot();
    }

    /**
     * {@inheritdoc}
     */
    function getNamespace()
    {
        return 'sites';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.site.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return 'Application\\Core\\Model\\Site';
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getContainer()->getService('translator')->trans('Sites');
    }

    /**
     * {@inheritdoc}
     */
    public function hasAddButton()
    {
        return $this->getHelper()->getUser()->isFounder();
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getContainer()->getService('translator')->trans('Add site');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($model->isNewRecord()) {
            return $translator->trans('Add site');
        }

        return $translator->trans('Edit the site “%title%”', array('%title%' => $model->name));
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        return array(
            array(
                'name'      => 'name',
                'label'     => $translator->trans('Site'),
                'class'     => 'large',
                'orderable' => true,
                'render'    => function ($row) {
                    return '<strong>' . $row->name . '</strong><div class="actions"><span>' . $row->actions . '</span></div>';
                }
            ),
            array(
                'name'   => 'domain',
                'label'  => $translator->trans('Domain'),
                'class'  => 'main',
                'render' => function ($site) {
                    /** @var \Application\Core\Model\Site $site */
                    return $site->permalink();
                }
            ),
            array(
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ),
        );
    }

    /**
     * {@inheritdoc}
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');

        $actions = array(
            array($this->container->getService('translator')->trans('Edit'), 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'),
        );

        $row->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $subject = $this->getSubject();

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        // Impostazioni principali
        $options = new FormConfig('sites', $subject, $this->getHelper()->getLocalization());

        $options->addBaseStructure('main', '');
        $options->addTab('content/main', $translator->trans('Main'));
        $options->addPanel('content/main/site');

        $options->addFields(
			'content/main/site',
			array(
                'name' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('Name'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'domain' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('Domain'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'content_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('content_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'images_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('images_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'images_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('images_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'uploads_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('uploads_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'uploads_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('uploads_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'themes_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('themes_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'themes_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('themes_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
            )
		);

        if ($this->getAction() === 'add') {
            $subject
                ->attachGetter('source_language_site', array($this, 'getSubjectSourceLanguageSiteCallback'))
                ->attachSetter('source_language_site', array($this, 'setSubjectSourceLanguageSiteCallback'))
            ;

            $options->addFields(
                'content/main/site',
                array(
                    'source_language_site'  => array(
                        'class'         => 'Application\\Core\\Form\\Field\\LanguageSiteField',
                        'label'         => $translator->trans('translation source'),
                        'attributes'    => array(
                            'required'      => 'required'
                        ),
                    ),
                )
            );
        }

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_form'), $options, array('subject' => $subject));
    }

    /**
     * Callback di tipo getter per ottenere il `source_language_site_id` dalla form.
     *
     * @access public
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @return int
     */
    public function getSubjectSourceLanguageSiteCallback(ModelSubject $subject)
    {
        return $this->source_language_site_id;
    }

    /**
     * Callback di tipo setter per impostare il `source_language_site_id` nella form.
     *
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @param                                   $value
     */
    public function setSubjectSourceLanguageSiteCallback(ModelSubject $subject, $value)
    {
        $this->source_language_site_id = $value;
    }

    /**
     * Restituisce il subject della form.
     *
     * @access protected
     * @return \Pongho\Form\Subject\ModelSubject
     */
    protected function getSubject()
    {
        $subject = new ModelSubject($this->getModel());

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        return array();
    }

    /**
     * Risponde alla callback `after_save`.
     *
     * Imposto i parametri base dopo la creazione del sito.
     *
     * @access public
     * @param \Pongho\Form\Form $form
     */
    public function afterSaveCallback(Form $form)
    {
        if ($this->getAction() === 'add') {
            /** @var \Application\Core\Model\Site $site_model */
            $site_model = $form->getSubject()->getModel();

            /** @var \Application\Core\Model\LanguageSite $source_language_site */
            $source_language_site = LanguageSite::find($this->source_language_site_id);

            /**
             * Aggiungo l’associazione tra sito e lingua di default
             *
             * @var \Application\Core\Model\LanguageSite $language_site
             */
            $language_site = LanguageSite::create(
                array(
                    'language_id' => $source_language_site->language_id,
                    'site_id'     => $site_model->id,
                    'path'        => '/',
                    'is_enabled'  => true,
                )
            );

            $site_model->default_language_id = $source_language_site->language_id;
            $site_model->save();

            // Copio le traduzioni da un sito all’altro
            foreach ($source_language_site->getTranslations() as $translation) {
                $attributes = $translation->attributes();

                $attributes['language_site_id'] = $language_site->id;

                Translation::create($attributes);
            }

            // Aggiungo la rotta degli utenti nei moduli
            $this->addModuleSite('Application\\Core\\Controller\\UserController', $language_site, $source_language_site);

            $this->getHelper()->notify(
                $this,
                'core.after_create_site',
                array(
                    'site_model'           => $site_model,
                    'language_site'        => $language_site,
                    'source_language_site' => $source_language_site,
                )
            );
        }
    }

    /**
     * @access public
     * @param string                               $controller_name
     * @param \Application\Core\Model\LanguageSite $language_site
     * @param \Application\Core\Model\LanguageSite $source_language_site
     * @return \Application\Core\Model\ModuleSite|null
     */
    public function addModuleSite($controller_name, LanguageSite $language_site, LanguageSite $source_language_site)
    {
        /** @var \Application\Core\Model\Module $user_module */
        $user_module = Module::first(
            array(
                'select'        => '`from`.*, lm.path, lm.name, lm.position, lm.is_enabled',
                'joins'         => 'LEFT JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = `from`.id',
                'conditions'    => array(
                    '`from`.controller = :controller AND lm.language_site_id = :language_site',
                    'controller'    => $controller_name,
                    'language_site' => $source_language_site->id,
                ),
            )
        );

        $module_site = null;

        if ($user_module) {
            /** @var \Application\Core\Model\ModuleSite $module_site */
            $module_site = ModuleSite::create(
                array(
                    'module_id'         => $user_module->id,
                    'site_id'           => $language_site->site_id,
                    'is_enabled'        => true,
                )
            );

            LanguageModule::create(
                array(
                    'language_site_id'  => $language_site->id,
                    'module_id'         => $user_module->id,
                    'path'              => $user_module->path,
                    'name'              => $user_module->name,
                    'position'          => $user_module->position,
                    'is_enabled'        => $user_module->is_enabled,
                )
            );
        }

        return $module_site;
    }
}
