<?php

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudController;
use Application\Core\Model\Account;
use Application\Core\Model\Group;
use Application\Core\Model\GroupUser;
use Pongho\Http\Exception\HttpException;

class GroupsController extends CrudController
{
    /**
     * {@inheritdoc}
     */
    public function displayArchive($page = 1)
    {
        /** @var Group[] $groups */
        $groups = Group::all(
            array(
                'order' => 'name',
            )
        );

        foreach ($groups as $group) {
            $actions = array(
                array('edit', 'href' => $this->url("/groups/edit/$group->id/"), 'class' => 'edit'),
                array('delete', 'href' => $this->url("/groups/delete/$group->id/"), 'class' => 'delete'),
            );

            $group->actions = $this->parseActions($actions);
        }

        $this->getHelper()->getHead()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(__DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css')));

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/groups_list.php')
            ->assignVars(
                array(
                    'rows'    => $groups,
                    'add_url' => $this->url('/groups/add/'),
                )
            );
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return 'Application\\Core\\Model\\Group';
    }

    /**
     * {@inheritdoc}
     */
    public function getTitle($model)
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        return $this->getAction() === 'add' ? $translator->trans('Add group') : sprintf($translator->trans('Edit group'), $model->name);
    }

    /**
     * {@inheritdoc}
     */
    public function getFields($model)
    {
        return $this->getHelper()->filter(
            $this,
            'admin.groups.filter_fields',
            array(
                array('name'),
                array('description', 'class' => 'Application\\Admin\\Form\\TextareaField')
            )
        );
    }

    /**
     * @param Group $group
     */
    protected function beforeSetTemplateCallback(Group $group)
    {
        $group_user_table = GroupUser::tableName();
        $options = array(
            'select'     => '`from`.*',
            'joins'      => "LEFT JOIN {$group_user_table} AS gu ON gu.user_id = `from`.id",
            'conditions' => array(
                '(gu.group_id <> :group OR gu.group_id IS NULL) AND `from`.id <> 1',
                'group' => $group->id
            ),
        );
        $available_users = Account::all($options);

        $group_users = $group->getUsers();

        // elenchi in json per la gestione delle tabelle via javascript
        $group_users_json = array();
        foreach ($group_users as $user) {
            $group_users_json[$user->user_id] = array(
                'user_id' => $user->user_id,
                'name'    => $user->user->name() ?: '&nbsp;',
                'company' => $user->user->company ?: '&nbsp;',
                'email'   => $user->user->email ?: '&nbsp;',
            );
        }

        $available_users_json = array();
        /** @var Account $user */
        foreach ($available_users as $user) {
            if (array_key_exists($user->id, $group_users_json)) {
                continue;
            }

            $available_users_json[$user->id] = array(
                'user_id' => $user->id,
                'name'    => $user->name() ?: '&nbsp;',
                'company' => $user->company ?: '&nbsp;',
                'email'   => $user->email ?: '&nbsp;',
            );
        }

        $users_json = json_encode(array('inGroup' => $group_users_json, 'noGroup' => $available_users_json));

        $url = $this->url("/groups/move/{$group->id}/");

        $js = <<<JS
(function($) {

	var users = $users_json,
		ingroup = $('#inGroup'),
		nogroup = $('#noGroup'),
		tpl = '<tr><td class="ckb"><input type="checkbox" name="users[{user_id}]" /></td><td class="main">{name}<span>{company}</span></td><td class="large">{email}</td></tr>';

	users.inGroup = users.inGroup || {};
	users.inGroup = users.inGroup.constructor == Array ? {} : users.inGroup;

	users.noGroup = users.noGroup || {};
	users.noGroup = users.noGroup.constructor == Array ? {} : users.noGroup;

	$.fn.outputTable = function(data, tpl) {
		var table = $(this),
			rows = '';

		$.each(data, function(index, row) {
			var code = tpl;

			$.each(row, function(field, value) {
				code = code.replace(new RegExp('{' + field + '}', 'g'), value);
			});

			rows += code;
		});

		table.find('.inner-table tbody').html(rows);
	};

	ingroup.outputTable(users.inGroup, tpl);
	nogroup.outputTable(users.noGroup, tpl);


	$("#group-remove").click(function(){
		var c = $(".inner-table input:checked", ingroup), movedUsers = [];

		if ( c.length )
		{
			c.each(function(){
				var name = $(this).attr('name'), id;

				// ricavo l'id
				id = name.substr(6, name.length - 7);

				movedUsers.push(id);

				// sposto la riga nella colonna di destra
				users.noGroup[id] = users.inGroup[id];
				delete users.inGroup[id];
			});

			updateTables({
				'direction'	: 'remove',
				'users'		: movedUsers
			});
		}
	});

	$("#group-insert").click(function(){
		var c = $(".inner-table input:checked", nogroup), movedUsers = [];

		if ( c.length )
		{
			c.each(function(){
				var name = $(this).attr('name');

				// ricavo l'id
				id = name.substr(6, name.length - 7);

				movedUsers.push(id);

				// sposto la riga nella colonna di sinistra
				users.inGroup[id] = users.noGroup[id];
				delete users.noGroup[id];
			});

			updateTables({
				'direction'	: 'insert',
				'users'		: movedUsers
			});
		}
	});

	function updateTables(movedData) {
		$.post('$url', movedData, function(d) {
			if (d.message === 'OK') {
				ingroup.outputTable(users.inGroup, tpl);
				nogroup.outputTable(users.noGroup, tpl);
			}
		});
	}

})(jQuery);
JS;

        $this->getHelper()->getBodyView()
            ->assignVars(
                array(
                    'users'           => $group_users,
                    'available_users' => $available_users,
                )
            );

        $this->getHelper()->getHead()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(__DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css')))
            ->addJavaScriptInline($js);
    }

    /**
     * @return \Pongho\Http\JsonResponse
     * @throws \ActiveRecord\Exceptions\ActiveRecordException
     */
    public function moveAction()
    {
        if ($this->getRequest()->getMethod() === 'POST') {
            if (!isset($_POST['direction']) || !isset($_POST['users']) || !is_array($_POST['users'])) {
                //Precondition Failed
                throw new HttpException(412);
            }

            $group_id = $this->getParameter('id');

            if (!$group_id) {
                throw new HttpException(404);
            }

            switch ($_POST['direction']) {
                case 'insert':
                    foreach ($_POST['users'] as $user) {
                        GroupUser::create(array('group_id' => $group_id, 'user_id' => $user));
                    }
                    break;

                case 'remove':
                    foreach ($_POST['users'] as $user) {
                        $user = GroupUser::find($group_id, $user);
                        $user->delete();
                    }
                    break;
            }

            return $this->getHelper()->displayJsonMessage('OK');
        }

        //Method Not Allowed
        throw new HttpException(405);
    }
}
