<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Command;

use Application\Core\I18n\Translator\PhpExtractor;
use Symfony\Component\Translation\Catalogue\DiffOperation;
use Symfony\Component\Translation\Catalogue\MergeOperation;
use Symfony\Component\Translation\Dumper\XliffFileDumper;
use Symfony\Component\Translation\Loader\MoFileLoader;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\MessageCatalogue;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Translation\Writer\TranslationWriter;

/**
 * A command that parse templates to extract translation messages and add them into the translation files.
 *
 * @author Michel Salib <michelsalib@hotmail.com>
 */
class TranslationUpdateCommand extends Command
{
    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('translation:update')
            ->setDefinition(array(
                    new InputArgument('locale', InputArgument::REQUIRED, 'The locale'),
                    new InputArgument('application', InputArgument::REQUIRED, 'The application where to load the messages'),
                    new InputOption(
                        'prefix', null, InputOption::VALUE_OPTIONAL,
                        'Override the default prefix', ''
                    ),
                    new InputOption(
                        'output-format', null, InputOption::VALUE_OPTIONAL,
                        'Override the default output format', 'xliff'
                    ),
                    new InputOption(
                        'dump-messages', null, InputOption::VALUE_NONE,
                        'Should the messages be dumped in the console'
                    ),
                    new InputOption(
                        'force', null, InputOption::VALUE_NONE,
                        'Should the update be done'
                    ),
                    new InputOption(
                        'no-backup', null, InputOption::VALUE_NONE,
                        'Should backup be disabled'
                    ),
                    new InputOption(
                        'clean', null, InputOption::VALUE_NONE,
                        'Should clean not found messages'
                    )
                ))
            ->setDescription('Updates the translation file')
            ->setHelp(<<<EOF
The <info>%command.name%</info> command extract translation strings from templates
of a given bundle. It can display them or merge the new ones into the translation files.
When new translation strings are found it can automatically add a prefix to the translation
message.

<info>php %command.full_name% --dump-messages en AcmeBundle</info>
<info>php %command.full_name% --force --prefix="new_" fr AcmeBundle</info>

EOF
            )
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        // check presence of force or dump-message
        if ($input->getOption('force') !== true && $input->getOption('dump-messages') !== true) {
            $output->writeln('<info>You must choose one of --force or --dump-messages</info>');

            return 1;
        }

        // check format

        /** @var TranslationWriter $writer */
        $writer = new TranslationWriter();
        $writer->addDumper('xliff', new XliffFileDumper());

        $supported_formats = $writer->getFormats();

        if (!in_array($input->getOption('output-format'), $supported_formats)) {
            $output->writeln('<error>Wrong output format</error>');
            $output->writeln('Supported formats are '.implode(', ', $supported_formats).'.');

            return 1;
        }

        // get bundle directory
        $application = ucfirst($input->getArgument('application'));
        $application_path = $translations_path = PONGHO_PATH . '/Application/' . $application;
        $translations_path = $application_path . '/Resources/translations';

        $locale = str_replace('_', '-', $input->getArgument('locale'));
        $output->writeln(sprintf('Generating "<info>%s</info>" translation files for "<info>%s</info>"', $locale, $application));

        // load any messages from templates
        $extracted_catalogue = new MessageCatalogue($locale);
        $output->writeln('Parsing templates');

        /** @var PhpExtractor $extractor */
        $extractor = new PhpExtractor();
        $extractor->setPrefix($input->getOption('prefix'));
        $extractor->extract($application_path, $extracted_catalogue);

        // load any existing messages from the translation files
        $current_catalogue = new MessageCatalogue($locale);
        $output->writeln('Loading translation files');

        /** @var TranslationLoader $loader */
        $loader = new TranslationLoader();
        $loader->addLoader('mo', new MoFileLoader());
        $loader->addLoader('xlf', new XliffFileLoader());
        $loader->loadMessages($translations_path, $current_catalogue);

        // process catalogues
        $operation = $input->getOption('clean')
            ? new DiffOperation($current_catalogue, $extracted_catalogue)
            : new MergeOperation($current_catalogue, $extracted_catalogue);

        // show compiled list of messages
        if ($input->getOption('dump-messages') === true) {
            foreach ($operation->getDomains() as $domain) {
                $output->writeln(sprintf("\nDisplaying messages for domain <info>%s</info>:\n", $domain));
                $newKeys = array_keys($operation->getNewMessages($domain));
                $allKeys = array_keys($operation->getMessages($domain));
                foreach (array_diff($allKeys, $newKeys) as $id) {
                    $output->writeln($id);
                }
                foreach ($newKeys as $id) {
                    $output->writeln(sprintf('<fg=green>%s</>', $id));
                }
                foreach (array_keys($operation->getObsoleteMessages($domain)) as $id) {
                    $output->writeln(sprintf('<fg=red>%s</>', $id));
                }
            }

            if ($input->getOption('output-format') == 'xliff') {
                $output->writeln('Xliff output version is <info>1.2</info>');
            }
        }

        if ($input->getOption('no-backup') === true) {
            $writer->disableBackup();
        }

        // save the files
        if ($input->getOption('force') === true) {
            $output->writeln('Writing files');
            $writer->writeTranslations(
                $operation->getResult(),
                $input->getOption('output-format'),
                array(
                    'path'           => $translations_path,
                    'default_locale' => 'it-IT',
                )
            );
        }

        return null;
    }
}
