<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Command;

use Application\Core\I18n\Translator\PhpExtractor;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Translation\Catalogue\MergeOperation;
use Symfony\Component\Translation\Loader\MoFileLoader;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\MessageCatalogue;
use Symfony\Component\Translation\Translator;

/**
 * Class TranslationDebugCommand
 * * Helps finding unused or missing translation messages in a given locale
 * and comparing them with the fallback ones.
 *
 * @author Florian Voutzinos <florian@voutzinos.com>
 */
class TranslationDebugCommand extends Command
{
    const MESSAGE_MISSING = 0;
    const MESSAGE_UNUSED = 1;
    const MESSAGE_EQUALS_FALLBACK = 2;

    /** @var Translator */
    protected $translator;

    /**
     * @param Translator $translator
     */
    public function __construct(Translator $translator)
    {
        $this->translator = $translator;

        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    public function configure()
    {
        $this
            ->setName('translation:debug')
            ->setDefinition(
                array(
                    new InputArgument('locale', InputArgument::REQUIRED, 'The locale'),
                    new InputArgument('application', InputArgument::REQUIRED, 'The application'),
                    new InputOption('domain', null, InputOption::VALUE_OPTIONAL, 'The messages domain'),
                    new InputOption('only-missing', null, InputOption::VALUE_NONE, 'Displays only missing messages'),
                    new InputOption('only-unused', null, InputOption::VALUE_NONE, 'Displays only unused messages'),
                )
            )
            ->setDescription('Displays translation messages informations')
            ->setHelp(
                <<<EOF
                The <info>%command.name%</info> command helps finding unused or missing translation
messages and comparing them with the fallback ones by inspecting the
templates and translation files of a given bundle.

You can display information about bundle translations in a specific locale:

<info>php %command.full_name% en AcmeDemoBundle</info>

You can also specify a translation domain for the search:

<info>php %command.full_name% --domain=messages en AcmeDemoBundle</info>

You can only display missing messages:

<info>php %command.full_name% --only-missing en AcmeDemoBundle</info>

You can only display unused messages:

<info>php %command.full_name% --only-unused en AcmeDemoBundle</info>

EOF
            );
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $locale = str_replace('_', '-', $input->getArgument('locale'));
        $domain = $input->getOption('domain');
        $application = ucfirst($input->getArgument('application'));

        $loader = new TranslationLoader();
        $loader->addLoader('mo', new MoFileLoader());
        $loader->addLoader('xlf', new XliffFileLoader());

        // Extract used messages
        $extracted_catalogue = new MessageCatalogue($locale);
        $extractor = new PhpExtractor();
        $extractor->extract(PONGHO_PATH . '/Application/' . $application . '/', $extracted_catalogue);

        // Load defined messages
        $current_catalogue = new MessageCatalogue($locale);
        $translations_path = PONGHO_PATH . '/Application/' . $application . '/Resources/translations';

        if (is_dir($translations_path)) {
            $loader->loadMessages($translations_path, $current_catalogue);
        }

        // Merge defined and extracted messages to get all message ids
        $mergeOperation = new MergeOperation($extracted_catalogue, $current_catalogue);
        $all_messages = $mergeOperation->getResult()->all($domain);

        if (null !== $domain) {
            $all_messages = array($domain => $all_messages);
        }

        // No defined or extracted messages
        if (empty($all_messages) || null !== $domain && empty($all_messages[$domain])) {
            $output_message = sprintf('<info>No defined or extracted messages for locale "%s"</info>', $locale);
            if (null !== $domain) {
                $output_message .= sprintf(' <info>and domain "%s"</info>', $domain);
            }
            $output->writeln($output_message);
            return;
        }

        // Load the fallback catalogues
        /** @var \Symfony\Component\Translation\MessageCatalogue[] $fallback_catalogues */
        $fallback_catalogues = array();

        foreach ($this->translator->getFallbackLocales() as $fallback_locale) {
            if ($fallback_locale === $locale) {
                continue;
            }
            $fallback_catalogue = new MessageCatalogue($fallback_locale);
            $loader->loadMessages($translations_path, $fallback_catalogue);
            $fallback_catalogues[] = $fallback_catalogue;
        }

        /** @var \Symfony\Component\Console\Helper\Table $table */
        $table = new Table($output);

        // Display header line
        $headers = array('State(s)', 'Id', sprintf('Message Preview (%s)', $locale));

        foreach ($fallback_catalogues as $fallback_catalogue) {
            $headers[] = sprintf('Fallback Message Preview (%s)', $fallback_catalogue->getLocale());
        }

        $table->setHeaders($headers);

        // Iterate all message ids and determine their state
        foreach ($all_messages as $domain => $messages) {
            foreach (array_keys($messages) as $message_id) {
                $value = $current_catalogue->get($message_id, $domain);
                $states = array();

                if ($extracted_catalogue->defines($message_id, $domain)) {
                    if (!$current_catalogue->defines($message_id, $domain)) {
                        $states[] = self::MESSAGE_MISSING;
                    }
                } elseif ($current_catalogue->defines($message_id, $domain)) {
                    $states[] = self::MESSAGE_UNUSED;
                }

                if (!in_array(self::MESSAGE_UNUSED, $states) && true === $input->getOption('only-unused')
                    || !in_array(self::MESSAGE_MISSING, $states) && true === $input->getOption('only-missing')
                ) {
                    continue;
                }

                foreach ($fallback_catalogues as $fallback_catalogue) {
                    if ($fallback_catalogue->defines($message_id, $domain) && $value === $fallback_catalogue->get($message_id, $domain)) {
                        $states[] = self::MESSAGE_EQUALS_FALLBACK;
                        break;
                    }
                }

                $row = array($this->formatStates($states), $this->formatId($message_id), $this->sanitizeString($value));

                foreach ($fallback_catalogues as $fallback_catalogue) {
                    $row[] = $this->sanitizeString($fallback_catalogue->get($message_id, $domain));
                }

                $table->addRow($row);
            }
        }

        $table->render();
        $output->writeln('');
        $output->writeln('<info>Legend:</info>');
        $output->writeln(sprintf(' %s Missing message', $this->formatState(self::MESSAGE_MISSING)));
        $output->writeln(sprintf(' %s Unused message', $this->formatState(self::MESSAGE_UNUSED)));
        $output->writeln(sprintf(' %s Same as the fallback message', $this->formatState(self::MESSAGE_EQUALS_FALLBACK)));
    }

    /**
     * @param $state
     * @return string
     */
    private function formatState($state)
    {
        if (self::MESSAGE_MISSING === $state) {
            return '<fg=red>x</>';
        }

        if (self::MESSAGE_UNUSED === $state) {
            return '<fg=yellow>o</>';
        }

        if (self::MESSAGE_EQUALS_FALLBACK === $state) {
            return '<fg=green>=</>';
        }

        return $state;
    }

    /**
     * @param array $states
     * @return string
     */
    private function formatStates(array $states)
    {
        $result = array();

        foreach ($states as $state) {
            $result[] = $this->formatState($state);
        }

        return implode(' ', $result);
    }

    /**
     * @param $id
     * @return string
     */
    private function formatId($id)
    {
        return sprintf('<fg=cyan;options=bold>%s</fg=cyan;options=bold>', $id);
    }

    /**
     * @param     $string
     * @param int $length
     * @return string
     */
    private function sanitizeString($string, $length = 40)
    {
        $string = trim(preg_replace('/\s+/', ' ', $string));

        if (function_exists('mb_strlen') && false !== $encoding = mb_detect_encoding($string)) {
            if (mb_strlen($string, $encoding) > $length) {
                return mb_substr($string, 0, $length - 3, $encoding) . '...';
            }
        } elseif (strlen($string) > $length) {
            return substr($string, 0, $length - 3) . '...';
        }

        return $string;
    }
}
