<?php

namespace Application\Core;

use Guzzle\Http\Client;
use Guzzle\Http\Exception\ClientErrorResponseException;
use Pongho\Http\JsonResponse;

class BaseApiController extends Controller
{
    /**
     * {@inheritdoc}
     */
    protected function handleInvalidResponse($response)
    {
        return new JsonResponse($response);
    }

    /**
     * Controlla se l'utente è loggato
     *
     * @return null|\Pongho\Http\JsonResponse
     */
    protected function checkLogIn()
    {
        // Utente loggato?
        if (!$this->getUser()->isLogged()) {
            $json = array(
                'error' => true,
                'message' => $this->getLocalization()->get('api_login_required'),
            );

            return new JsonResponse($json, array(), 401);
        }

        // Permesso consentito?
        if (!$this->getUser()->hasPermit('admin')) {
            $json = array(
                'error' => true,
                'message' => $this->getLocalization()->get('api_admin_permit_required'),
            );

            return new JsonResponse($json, array(), 401);
        }

        return null;
    }

    /**
     * Consente l'accesso alle API verificando la validità del token di accesso
     *
     * @param string $app_secret
     * @return null|\Pongho\Http\JsonResponse
     */
    protected function checkAuth($app_secret)
    {
        if (is_labs()) {
            $auth_host = 'http://labs.metaline.it';
            $auth_path = '/pongho/auth/check_token';
        } else {
            $auth_host = 'https://serverline.metaline.it';
            $auth_path = '/auth/check_token';
        }

        $client = new Client($auth_host);

        $options = array(
            'query' => array(
                'ip' => $this->getRequest()->getClientIp(),
                'app' => $app_secret,
            ),
            'auth' => array($this->getRequest()->getUser(), $this->getRequest()->getPassword(), 'Basic'),
        );

        $request = $client->get($auth_path, array(), $options);

        try {
            $request->send();
            $response = $request->getResponse();
        } catch (ClientErrorResponseException $e) {
            $response = $e->getResponse();
        }

        if ($response->isError()) {
            return new JsonResponse($response->json(), array(), $response->getStatusCode());
        }

        return null;
    }
}
