<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Application;

use Application\Core\Model\Manager\ApplicationManagerInterface;
use Application\Core\Model\Manager\Exception\ApplicationManagerException;
use Application\Core\Model\Manager\Exception\ApplicationManagerKernelException;
use Pongho\DependencyInjection\Container;

/**
 * Class Loader
 */
class Loader implements LoaderInterface
{
    /**
     * @var ApplicationManagerInterface
     */
    protected $manager;

    /**
     * @var \Pongho\DependencyInjection\Container
     */
    protected $container;

    /**
     * @param ApplicationManagerInterface $manager
     * @param Container                   $container
     */
    public function __construct(ApplicationManagerInterface $manager, Container $container)
    {
        $this->manager = $manager;
        $this->container = $container;
    }

    /**
     * {@inheritdoc}
     */
    public function getKernelClass($app_name)
    {
        return 'Application\\' . $app_name . '\\Kernel';
    }

    /**
     * {@inheritdoc}
     */
    public function kernelExists($app_name)
    {
        return class_exists($this->getKernelClass($app_name));
    }

    /**
     * {@inheritdoc}
     */
    public function load()
    {
        // carico le applicazioni
        foreach ($this->manager->allEnabled() as $app) {
            try {
                $kernel_class = $this->getKernelClass($app->getName());

                if (!$this->kernelExists($app->getName())) {
                    throw new ApplicationManagerKernelException(sprintf('The kernel for application "%s" not exists.', $app->getName()));
                }

                if (!is_subclass_of($kernel_class, 'Pongho\\Core\\Plugin', true)) {
                    throw new ApplicationManagerException(sprintf('Class %s not extends %s!', $kernel_class, '\\Pongho\\Core\\Plugin'));
                }

                /** @var \Pongho\Core\Plugin $application */
                $application = new $kernel_class($this->container);

                if ($application->isRunnable()) {
                    $application->load();
                }
            } catch (\Exception $e) {
                if ($this->container->getService('kernel')->isDebug()) {
                    throw new \RuntimeException(sprintf(
                        'Application "%s" thrown an exception with message: %s',
                        $app->getName(),
                        $e->getMessage()
                    ));
                }
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getKernel($app_name)
    {
        // Creo il nome del kernel
        $kernel_class = $this->getKernelClass($app_name);

        if (!class_exists($kernel_class)) {
            throw new ApplicationManagerKernelException(sprintf('The kernel for application "%s" not exists.', $app_name));
        }

        if (!defined("{$kernel_class}::VERSION")) {
            throw new ApplicationManagerKernelException(sprintf('The application "%s" kernel lacks the version.', $app_name));
        }

        return new $kernel_class($this->container);
    }
}
