<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core;

use Application\Core\DependencyInjection\AddressProvider;
use Application\Core\DependencyInjection\AdminBarServiceProvider;
use Application\Core\DependencyInjection\AdminSidebarServiceProvider;
use Application\Core\DependencyInjection\ApplicationServiceProvider;
use Application\Core\DependencyInjection\CachingServiceProvider;
use Application\Core\DependencyInjection\ConsoleCommandProvider;
use Application\Core\DependencyInjection\CoreServiceProvider;
use Application\Core\DependencyInjection\DbServiceProvider;
use Application\Core\DependencyInjection\LoggerServiceProvider;
use Application\Core\DependencyInjection\MailerServiceProvider;
use Application\Core\DependencyInjection\ModelManagerProvider;
use Application\Core\DependencyInjection\NotificationServiceProvider;
use Application\Core\DependencyInjection\RoutingServiceProvider;
use Application\Core\DependencyInjection\ShortcodeServiceProvider;
use Application\Core\DependencyInjection\TemplatingServiceProvider;
use Application\Core\DependencyInjection\TranslationServiceProvider;
use Application\Core\DependencyInjection\UserServiceProvider;
use Pongho\Core\Kernel;
use Pongho\EventDispatcher\Event;
use Pongho\Http\Request;
use Pongho\Http\Response;

/**
 * Estende il Kernel di Pongho per aggiungere il controllo sulle applicazioni e i plugin.
 */
class AppKernel extends Kernel
{
    protected function handleRaw(Request $request)
    {
        $this->registerComponents();

        /**
         * Lancio un evento prima della normale gestione della richiesta.
         *
         * @var \Pongho\EventDispatcher\EventDispatcher $dispatcher
         */
        $event = new Event($this, 'core.replace_handle');
        $dispatcher = $this->container->getService('event_dispatcher');
        $dispatcher->notifyUntil($event);

        if ($event->isProcessed() && ($response = $event->getReturnValue()) instanceof Response) {
            return $response;
        }

        return parent::handleRaw($request);
    }

	/**
	 * {@inheritdoc}
	 */
	public function handleConsole(Request $request)
	{
        parent::handleConsole($request);

        $this->registerComponents();

        return $this;
	}

	/**
	 * {@inheritdoc}
	 */
	protected function registerComponents()
	{
		// Attenzione all'ordine
		$this->container->register(new CachingServiceProvider());
		$this->container->register(new DbServiceProvider());
		$this->container->register(new UserServiceProvider());
		$this->container->register(new ModelManagerProvider());
		$this->container->register(new RoutingServiceProvider());
		$this->container->register(new TranslationServiceProvider());
		$this->container->register(new ConsoleCommandProvider());
		$this->container->register(new AddressProvider());
		$this->container->register(new ShortcodeServiceProvider());
		$this->container->register(new TemplatingServiceProvider());
		$this->container->register(new MailerServiceProvider());
		$this->container->register(new LoggerServiceProvider());
        $this->container->register(new NotificationServiceProvider());
        $this->container->register(new AdminSidebarServiceProvider());
        $this->container->register(new CoreServiceProvider());

		$this->registerApplications();
        $this->container->register(new AdminBarServiceProvider());
		$this->registerPlugins();

        /** @var \Pongho\EventDispatcher\EventDispatcher $event_dipatcher */
        $event_dipatcher = $this->container->getService('event_dispatcher');
        $event_dipatcher->notify(new Event($this, 'after_register_components'));
	}

	/**
	 * {@inheritdoc}
	 */
	protected function getDefaultParameters()
	{
		return array_merge(parent::getDefaultParameters(), array(
			'cache.enabled'					=> true,
			'cache.cache_class'				=> null,

			'container.class'				=> 'Application\\Core\\Container',

            'controller_helper_class'       => 'Application\\Core\\Utilities\\ControllerHelper',

            'mailer.mailer_class'           => 'Application\\Core\\Utilities\\Mailer',
            'mailer.transport_class'        => 'Swift_SmtpTransport',
            'mailer.transport_params'       => array('batmailrelay.metaline.it', 25),

			'orm.config_path'				=> CONFIG_PATH . '/db.php',

			'shortcode.shortcode_class'		=> 'Application\\Core\\Utilities\\Shortcode',

			'template.theme_class'			=> 'Pongho\\Template\\Theme',
			'template.html_class'			=> 'Application\\Core\\Utilities\\Html',

			'site_id'						=> 1,
		));
	}



    /**
     * Registra le applicazioni.
     *
     * @access private
     */
    private function registerApplications()
    {
		$this->container->register(new ApplicationServiceProvider());

		$this->container->getService('application_loader')->load();
	}

	/**
	 * Registra i plugin.
	 *
	 * @access private
	 */
	private function registerPlugins()
	{
		foreach ( glob(PLUGINS_PATH . '/*') as $path )
		{
            $class_name = null;
            $plugin_name = null;

			if ( is_file($path) )
			{
				$filename = basename($path, '.php');

				if ( substr($filename, 0, 1) === '_' || substr($path, -4) !== '.php' )
				{
					continue;
				}

				$plugin_name = $filename;
				$class_name = 'Plugins\\' . $filename;
			}
			else if ( is_dir($path) )
			{
				$dirname = basename($path);
				$path .= '/' . $dirname . '.php';

				if ( !file_exists($path) )
				{
					continue;
				}

				$plugin_name = $dirname;
				$class_name = 'Plugins\\' . $dirname . '\\' . $dirname;
			}

            if ($class_name === null) {
                continue;
            }

			try
			{
				require $path;

				if ( !class_exists($class_name) )
				{
					continue;
				}

                /** @var \Pongho\Core\Plugin $plugin */
				$plugin = new $class_name($this->getContainer());

				if ( $plugin->isRunnable() )
				{
					$plugin->load();
				}
			}
			catch ( \Exception $e )
			{
				if ( $this->isDebug() )
				{
					throw new \RuntimeException(sprintf('Plugin "%s" thrown an exception with message: %s', $plugin_name, $e->getMessage()));
				}
			}
		}
	}
}
