<?php

namespace Application\Core\Utilities;

require PONGHO_PATH . '/vendor/swiftmailer/swiftmailer/lib/swift_init.php';

use Application\Core\Entity\SiteInterface;
use Pongho\Core\LocalizationInterface;
use Swift_Image;
use Swift_Mailer;
use Swift_Message;
use Swift_Plugins_LoggerPlugin;
use Swift_Plugins_Loggers_ArrayLogger;

/**
 * Mailer.
 *
 * Questa classe offre un sistema semplificato per l'invio delle e-mail
 * appoggiandosi alla libreria SwiftMailer.
 *
 * @link http://swiftmailer.org/
 *
 * Esempio di utilizzo:
 * <code>
 *   $from = 'mario@rossi.it';
 *   $to = 'paolo@verdi.it';
 *   $subject = 'Oggetto della e-mail';
 *   $content = 'Ciao Paolo, questo è il contenuto della mia e-mail!';
 *   Mailer::send($from, $to, $subject, $content);
 * </code>
 *
 * Per quanto riguarda le variabili $from e $to, è possibile usare anche un
 * array in modo da indicare anche il nome cui fa riferimento l'indirizzo
 * e-mail. La variabile $to può contenere più nominativi.
 *
 * E' inoltre possibile creare del contenuto partendo da un template e dalle
 * variabili da sostituire in esso:
 * <code>
 *   $content = Mailer::getContent('path/to/template.php', array('NAME' => 'Mario Rossi'));
 *   Mailer::send($from, $to, $subject, $content);
 * </code>
 * All'interno del template la variabile 'NAME' deve essere indicata tra
 * parentesi graffe:
 * <code>
 *   Ciao {NAME}, questa è una notifica inviata da Pongho.
 * </code>
 *
 * L'estensione del template non ha importanza per la classe, ma per gli
 * standard del codice di Pongho è preferibile usare l'estensione php.
 */
class Mailer
{
	/**
	 * Sito di riferimento per la configurazione.
	 *
	 * @access protected
	 * @var SiteInterface
	 */
	protected $site;

	/**
	 * Linguaggio per le traduzioni.
	 *
	 * @access protected
	 * @var LocalizationInterface
	 */
	protected $language;

    /**
     * Transport per l’invio delle e-mail.
     *
     * @access protected
     * @var \Swift_Transport
     */
    protected $transport;

    /**
     * @access
     * @var \Swift_Mime_ContentEncoder
     */
    protected $encoder;

    /**
     * Costruttore.
     *
     * @param SiteInterface         $site
     * @param LocalizationInterface $language
     */
	public function __construct(SiteInterface $site, LocalizationInterface $language)
	{
		$this->site = $site;
		$this->language = $language;
	}

    /**
     * Permette di impostare il Transport per la spedizione della mail.
     *
     * @param \Swift_Transport $transport
     * @return $this
     */
    public function setTransport(\Swift_Transport $transport)
    {
        $this->transport = $transport;

        return $this;
    }

    /**
     * Restituisce il Transport.
     *
     * @return \Swift_Transport
     */
    public function getTransport()
    {
        if ($this->transport === null) {
            $this->transport = \Swift_SendmailTransport::newInstance();
        }

        return $this->transport;
    }

    /**
     * Invia una e-mail indicando alcuni parametri base.
     * Per scopi più complessi si consiglia di fare affidamento
     * alla libreria SwiftMailer.
     *
     * Mittente e destinatario possono essere anche un array del tipo:
     * <code>
     *   $from = array('mario@rossi.it' => 'Mario Rossi');
     *   $to = array('paolo@verdi.it' => 'Paolo Verdi', 'alfredo@bianchi.it' => 'Alfredo Bianchi');
     * </code>
     *
     * Il contenuto ci si aspetta sia in formato HTML.
     * Verrà creata anche una versione testuale partendo da quella inserita.
     *
     * @access public
     * @param mixed  $from
     * @param mixed  $to
     * @param string $subject
     * @param string $content
     * @param mixed  $reply_to
     * @param array  $attachments
     * @return boolean
     * @throws \RuntimeException
     */
	public function send($from, $to, $subject, $content, $reply_to = null, array $attachments = array())
	{
		// Preparo il messaggio
		$message = $this->prepareMessage($from, $to, $subject, $reply_to, $attachments);

		// Inserisco il logo nel contenuto dell’e-mail.
		if ( strpos($content, '{COMPANY_LOGO}') )
		{
            $logo_path = $this->site->getThemesPath('images/logo-email.png');

			if ( !file_exists($logo_path) )
			{
				throw new \RuntimeException(sprintf('Logo not found on path "%s"', $logo_path));
			}

			$content = str_replace('{COMPANY_LOGO}', $message->embed(Swift_Image::fromPath($logo_path)), $content);
		}

		// Aggiungo il contenuto
		$message->setBody($content, 'text/html')
			->addPart($this->getPlain($content), 'text/plain');

		return $this->sendMessage($message);
	}

	/**
	 * Invia un messaggio in testo semplice.
	 *
	 * Simile a send() (@see Mailer::send()), ma per le e-mail di solo testo.
	 *
	 * @access public
	 * @param mixed $from
	 * @param mixed $to
	 * @param string $subject
	 * @param string $content
	 * @param mixed $reply_to
	 * @param array $attachments
	 * @return boolean
	 */
	public function sendPlain($from, $to, $subject, $content, $reply_to = null, array $attachments = array())
	{
		// Preparo il messaggio
		$message = $this->prepareMessage($from, $to, $subject, $reply_to, $attachments)
			->setBody($content, 'text/plain');

		return $this->sendMessage($message);
	}

	/**
	 * Metodo interno per la preparazione del messaggio.
	 *
	 * @access protected
	 * @param mixed $from
	 * @param mixed $to
	 * @param string $subject
	 * @param mixed $reply_to
	 * @param array $attachments
	 * @return Swift_Message
	 */
	protected function prepareMessage($from, $to, $subject, $reply_to = null, array $attachments = array())
	{
		$message = Swift_Message::newInstance()
			->setSubject($subject)
			->setFrom($this->prepareEmail($from))
			->setTo($this->prepareEmail($to));

		// Reply-To
		if ( $reply_to !== null )
		{
			$message->setReplyTo($reply_to);
		}

		// Allegati
		foreach ( $attachments as $attachment )
		{
			$message->attach($attachment);
		}

		return $message;
	}

    /**
     * Metodo interno per l’invio del messaggio.
     *
     * @access protected
     * @param Swift_Message $message
     * @return boolean
     * @throws \RuntimeException
     */
	protected function sendMessage(Swift_Message $message)
	{
        $transport = $this->getTransport();
		$mailer = Swift_Mailer::newInstance($transport);

        $message->setEncoder(\Swift_Encoding::get8BitEncoding());

		// Log
		$logger = new Swift_Plugins_Loggers_ArrayLogger();
		$mailer->registerPlugin(new Swift_Plugins_LoggerPlugin($logger));

		if ( !$mailer->send($message) )
		{
			error_log($logger->dump(), 1, 'log@metaline.it');

			throw new \RuntimeException(sprintf('Could not send the email "%s"!', $message->getSubject()));
		}

		return true;
	}

    /**
     * Restituisce la versione testuale di una stringa HTML.
     *
     * @access public
     * @param string $html
     * @return string
     */
	public function getPlain($html)
	{
		$plain = preg_replace('#^(.*?)<body(.*?)>(.*?)</body>(.*?)$#si', '$3', $html);
		$plain = str_replace("\t", '', strip_tags($plain));

		return $plain;
	}

	/**
	 * Carica il contenuto di un template da file.
	 *
	 * @access public
	 * @param string $template_path
	 * @param array $vars
	 * @return string
	 */
	public function content($template_path, array $vars = array())
	{
		// Carico il contenuto
		$content = $this->getContentFromTemplate($template_path);

		if ( empty($content) )
		{
			return '';
		}

		return $this->parse($content, $vars);
	}

    /**
     * Dato un template ne sostituisce le variabili per l'inserimento nel corpo della e-mail.
     *
     * Per la traduzione delle chiavi si basa sulla lingua corrente del controller che contiene il modulo di contatto.
     *
     * @access public
     * @param string $string
     * @param array  $vars
     * @return string
     */
    public function parse($string, array $vars = array())
    {
        if (empty($string)) {
            return '';
        }

        $lang = $this->language;
        $site = $this->site;

        // Aggiungo le variabili di default
        $vars = array_merge($this->getDefaultVars(), array_change_key_case($vars, CASE_UPPER));

        $replace_language = function (array $matches) use ($lang) {
            return $lang->get($matches[1]);
        };

        $replace_site_config = function (array $matches) use ($site) {
            return $site->getOption($matches[1]);
        };

        $replace_var = function (array $matches) use ($vars) {
            return isset($vars[$matches[1]]) ? $vars[$matches[1]] : $matches[0];
        };

        // Sostituisco le variabili
        $string = str_replace('\\', '\\\\', $string);
        $string = str_replace('\'', '\\\'', $string);

        $string = preg_replace_callback("#{L_(.*?)}#", $replace_language, $string);
        $string = preg_replace_callback("#%L_(.*?)%#", $replace_language, $string);

        $string = str_replace("{C_site_name}", $this->site->getName(), $string);
        $string = str_replace("%C_site_name%", $this->site->getName(), $string);

        $string = preg_replace_callback("#{C_(.*?)}#", $replace_site_config, $string);
        $string = preg_replace_callback("#%C_(.*?)%#", $replace_site_config, $string);

        $string = preg_replace_callback("#{(.*?)}#", $replace_var, $string);
        $string = preg_replace_callback("#%(.*?)%#", $replace_var, $string);

        $string = str_replace('\\\\', '\\', $string);
        $string = str_replace('\\\'', '\'', $string);

        return $string;
    }

    /**
     * Carica il contenuto del file indicato.
     *
     * @access protected
     * @param string $template_path
     * @throws \UnexpectedValueException
     * @return string
     */
	protected function getContentFromTemplate($template_path)
	{
		if ( !file_exists($template_path) )
		{
			throw new \UnexpectedValueException(sprintf('Email template "%s" not exists!', $template_path));
		}

		$content = trim(file_get_contents($template_path));

		return $content;
	}

	/**
	 * Restituisce le variabili di default da usare nelle email.
	 *
	 * @access protected
	 * @return array
	 */
	protected function getDefaultVars()
	{
		// Autore del sito
		if ( $this->site->getOption('author_website') )
		{
			$author = '<a href="' . $this->site->getOption('author_website') . '" style="' . $this->site->getOption('email_link_style') . '">' . $this->site->getOption('author_name') . '</a>';
		}
		else
		{
			$author = $this->site->getOption('author_name');
		}

		return array(
			'COMPANY_SITE'			=> absolute_url(''),
			'EMAIL_PRIVACY'			=> sprintf($this->language->get('email_privacy'), $this->site->getOption('company_name')),
			'EMAIL_COPYRIGHT'		=> sprintf($this->language->get('email_copyright'), $author),
			'EMAIL_DATE_SEND'		=> sprintf($this->language->get('email_date_send'), date('d/m/Y')),
		);
	}

    /**
     * Notifica gli errori all’amministratore.
     *
     * @access public
     * @param string $subject
     * @param string $error
     * @param string $notes
     * @return bool
     */
    public function errorNotify($subject, $error, $notes = '')
    {
        $domain = $this->site->getDomain();

        if ($error instanceof \Exception) {
            $error = nl2br($error->__toString());
        }

        $body = <<<HTML
<html>
<body>

<table style="border-collapse: collapse; width: 100%;">
    <tr>
        <th style="border-bottom: 1px solid #ccc; font-family: Helvetica, Arial, sans-serif; font-size: 14px; font-weight: bold; padding: 5px; vertical-align: top;">Domain:</th>
        <td style="border-bottom: 1px solid #ccc; font-family: Helvetica, Arial, sans-serif; font-size: 14px;padding: 5px; vertical-align: top;">{$domain}</td>
    </tr>
    <tr>
        <th style="border-bottom: 1px solid #ccc; font-family: Helvetica, Arial, sans-serif; font-size: 14px;font-weight: bold; padding: 5px; vertical-align: top;">Error:</th>
        <td style="border-bottom: 1px solid #ccc; font-family: Helvetica, Arial, sans-serif; font-size: 14px;padding: 5px; vertical-align: top;">{$error}</td>
    </tr>
    <tr>
        <th style="font-family: Helvetica, Arial, sans-serif; font-size: 14px;font-weight: bold; padding: 5px; vertical-align: top;">Notes:</th>
        <td style="font-family: Helvetica, Arial, sans-serif; font-size: 14px;padding: 5px; vertical-align: top;">{$notes}</td>
    </tr>
</table>

</body>
</html>
HTML;

        $message = new Swift_Message($subject, $body, 'text/html');

        $message->setFrom('no-reply@metaline.it');
        $message->setTo('log@metaline.it');

        return $this->sendMessage($message);
    }

    /**
     * @param $email
     * @return array|mixed
     */
	protected function getEmailAddress($email)
	{
		if ( is_array($email) )
		{
			$email = array_keys($email);

			return reset($email);
		}

		return $email;
	}

    /**
     * @param $email
     * @return mixed
     */
	protected function prepareEmail($email)
	{
		if ( is_array($email) )
		{
			foreach ( $email as $address => &$name )
			{
				$name = str_replace(array('"'), '', $name);
			}
		}

		return $email;
	}
}
