<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core;

use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;
use Pongho\Http\Session;
use Pongho\Utilities\DateTime;

/**
 * Rappresenta l’utente corrente.
 */
class User
{
    /**
     * Modello Account.
     *
     * @var AccountInterface
     */
    private $data;

    /**
     * @var Session
     */
    protected $session;

    /**
     * Costruttore.
     *
     * @param Session               $session
     */
    public function __construct(Session $session)
    {
        $this->session = $session;

        $this->setData();
    }

    /**
     * Cerca l’utente corrente e crea l’istanza con i dati in database.
     */
    private function setData()
    {
        $user_id = (int) $this->getSession()->user_id;

        if ($user_id <= 0) {
            $user_id = Account::ANONYMOUS;
        }

        if (($this->data = Account::find($user_id)) === null || $this->data->isActive() === false) {
            $this->data = Account::find(Account::ANONYMOUS);
        }

        $this->getSession()->user_id = $this->data->getId();
    }

    /**
     * Restituisce il modello dell’account legato all’utente corrente.
     *
     * @return AccountInterface
     */
    public function getAccount()
    {
        return $this->data;
    }

    /**
     * Restituisce un attributo del modello account dell’utente corrente.
     *
     * @param string $name
     * @return mixed
     */
    public function __get($name)
    {
        try {
            return $this->data->$name;
        } catch (\Exception $e) {
            return $name;
        }
    }

    /**
     * Esegue il login.
     *
     * @param string  $username_or_email
     * @param string  $password
     * @param boolean $autologin
     * @param array   $errors
     * @return boolean
     */
    public function login(&$username_or_email, &$password, $autologin, &$errors)
    {
        $lang = $this->getLanguage();

        // Campo nome utente o email obbligatorio
        if (empty($username_or_email)) {
            $errors['username_or_email'] = $lang->get('login_required_username_or_email');
            $username_or_email = '';

            return false;
        }

        // Campo password obbligatorio
        if (empty($password)) {
            $errors['password'] = $lang->get('login_required_password');
            $password = '';

            return false;
        }

        if (preg_match(REGEXP_VALIDATE_EMAIL, $username_or_email)) {
            // Ricavo l'account dall'indirizzo e-mail
            $userdata = Account::findByEmail($username_or_email);

            // Nessun account trovato
            if ($userdata === null) {
                $errors['username_or_email'] = $lang->get('login_email_not_found');
                $username_or_email = '';

                return false;
            }
        } else {
            // Ricavo l'account dal nome utente
            $userdata = Account::findByUsername($username_or_email);

            // Nessun account trovato
            if ($userdata === null) {
                $errors['username_or_email'] = $lang->get('login_username_not_found');
                $username_or_email = '';

                return false;
            }
        }

        /** @var Account $userdata */

        // Confronto la password dell'account con quella passata
        if ($userdata->getPassword() != $password) {
            $errors['password'] = $lang->get('login_password_not_valid');
            $password = '';

            return false;
        }

        // Utente attivo?
        if (!$userdata->isActive()) {
            $errors['active'] = $lang->get('login_user_not_active');
            $username = '';
            $password = '';

            return false;
        }

        // Aggiorna la data di ultimo login
        $userdata->last_login_at = new DateTime();
        $userdata->save();

        // Inizializzo l'utente
        $this->data = $userdata;
        $this->getSession()->user_id = $this->data->getId();

        return true;
    }

    /**
     * Forza il login all’account passato per gestire l’accesso automatico dopo alcune operazioni particolari.
     *
     * Ovviamente, **DA USARE CON CAUTELA**!
     *
     * @param AccountInterface $account
     * @return self
     */
    public function forceLogin(AccountInterface $account)
    {
        $this->data = $account;
        $this->getSession()->user_id = $account->getId();

        return $this;
    }

    /**
     * Esegue il logout dell’utente.
     *
     * @param boolean $continue Indica se dopo il logout lo script continua, oppure esce.
     * @return boolean
     */
    public function logout($continue = false)
    {
        $this->getSession()->destroy();

        if ($continue) {
            $this->data = Account::find(Account::ANONYMOUS);
        }
    }

    /**
     * Modifica i dati dell’utente corrente.
     *
     * @param array $data
     * @return boolean
     */
    public function edit(array $data = array())
    {
        $this->data->update_attributes($data);

        if ($this->data->save()) {
            return true;
        }

        return $this->data->errors;
    }

    /**
     * Restituisce il nome dell'utente.
     *
     * @return string
     */
    public function name()
    {
        return $this->data->name();
    }

    /**
     * Restituisce true se l’utente è loggato.
     *
     * @return boolean
     */
    public function isLogged()
    {
        return ($this->data->getId() !== Account::ANONYMOUS);
    }

    /**
     * Restituisce true se l’utente è amministratore.
     *
     * @return boolean
     */
    public function isAdmin()
    {
        return $this->data->isAdmin();
    }

    /**
     * Restituisce true se l’utente è founder.
     *
     * @return boolean
     */
    public function isFounder()
    {
        return $this->data->isFounder();
    }

    /**
     * Restituisce true se l’utente ha il permesso indicato.
     *
     * @param string $key
     * @return boolean
     */
    public function hasPermit($key)
    {
        return $this->data->hasPermit($key);
    }

    /**
     * Verifica se l’utente appartiene ad un gruppo.
     *
     * È possibile specificare anche una lista di gruppi passando come argomento un array.
     * Il gruppo può essere sia istanza di Group che l'ID del gruppo.
     *
     * @param mixed $groups
     * @return self
     */
    public function inGroup($groups)
    {
        return $this->data->inGroup($groups);
    }

    /**
     * Restituisce il gestore della sessione.
     *
     * @return \Pongho\Http\Session
     */
    protected function getSession()
    {
        return $this->session;
    }

    /**
     * Restituisce il gestore della localizzazione.
     *
     * @return Localization
     */
    protected function getLanguage()
    {
        // Problema di dipendenze circolari, "language" dipende da "user", non posso passargli il servizio
        return AppKernel::instance()->getContainer()->get('language');
    }
}
