<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Showcase
 */

namespace Application\Core\Tests\Utilities;

use Application\Core\Utilities\Shortcode;

class ShortcodeTest extends \PHPUnit_Framework_TestCase
{
    public function testAttributesTags()
    {
        $shortcode = new Shortcode();

        $shortcode->add(
            'foo',
            function ($arguments, $content, $tag_name) {
                return isset($arguments['bar']) ? $arguments['bar'] : 'not-found';
            }
        );

        $this->assertEquals(
            $shortcode->parse("[foo bar='foobar' /] [foo bar='barfoo' /]"),
            'foobar barfoo'
        );

        $this->assertEquals(
            $shortcode->parse('[foo bar="foobar" /] [foo bar="barfoo" /]'),
            'foobar barfoo'
        );

        $this->assertEquals(
            $shortcode->parse('[foo bar="foobar" /]'),
            'foobar'
        );

        $this->assertEquals(
            $shortcode->parse('[foo foo="foobar" /]'),
            'not-found'
        );

        $this->assertEquals(
            $shortcode->parse('[foo bar="foobar" /] ' . "\n" . ' [foo bar="barfoo" /]'),
            "foobar \n barfoo"
        );
    }

    public function testContentTags()
    {
        $shortcode = new Shortcode();

        $shortcode->add(
            'foo',
            function ($arguments, $content, $tag_name) {
                return empty($content) ? 'no-content' : $content;
            }
        );

        $this->assertEquals(
            $shortcode->parse('Lorem ipsum [foo]dolor[/foo] sit amet, [bar]consectetur[/bar] adipiscing elit.'),
            'Lorem ipsum dolor sit amet, [bar]consectetur[/bar] adipiscing elit.'
        );

        $this->assertEquals(
            $shortcode->parse('Lorem ipsum [foo][/foo] sit amet, [bar]consectetur[/bar] adipiscing elit.'),
            'Lorem ipsum no-content sit amet, [bar]consectetur[/bar] adipiscing elit.'
        );
    }

    public function testAttributesAndContentTags()
    {
        $shortcode = new Shortcode();

        $shortcode->add(
            'foo',
            function ($arguments, $content, $tag_name) {
                if (empty($content) || !isset($arguments['bar'])) {
                    return '';
                }

                return sprintf('[%1$s]%2$s[/%1$s]', $arguments['bar'], $content);
            }
        );

        $this->assertEquals(
            $shortcode->parse('Lorem ipsum [foo bar="foobar"]dolor[/foo] sit amet, [bar]consectetur[/bar] adipiscing elit.'),
            'Lorem ipsum [foobar]dolor[/foobar] sit amet, [bar]consectetur[/bar] adipiscing elit.'
        );

        $this->assertEquals(
            $shortcode->parse('Lorem ipsum [foo][/foo] sit amet, [bar]consectetur[/bar] adipiscing elit.'),
            'Lorem ipsum  sit amet, [bar]consectetur[/bar] adipiscing elit.'
        );
    }
}
