<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Mailer
{
    if (!function_exists('Application\\Core\\Mailer\\absolute_url')) {
        /**
         * @param string $url
         * @return string
         */
        function absolute_url($url) {
            return $url;
        }
    }
}

namespace Application\Core\Tests\Mailer
{
    use Application\Core\Localization;
    use Application\Core\Mailer\MailerHelper;
    use Application\Core\Model\Site;

    /**
     * Class MailerHelperTest
     */
    class MailerHelperTest extends \PHPUnit_Framework_TestCase
    {
        /**
         * @dataProvider parseEmailProvider
         * @param $email
         * @param $vars
         * @param $result
         */
        public function testParseEmails($email, $vars, $result)
        {
            $helper = new MailerHelper($this->getSiteMock(), $this->getLocalizationMock());

            $parsed = $helper->parseEmails($email, $vars);

            $this->assertEquals($result, $parsed);
        }

        /**
         * @return array
         */
        public function parseEmailProvider()
        {
            return array(
                'standard case' => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'NAME'  => 'Foo',
                        'EMAIL' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Foo',
                    )
                ),
                'lowercase vars' => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'name'  => 'Foo',
                        'email' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Foo',
                    )
                ),
                'multiple emails' => array(
                    '%NAME% <%EMAIL%>, %COMPANY_NAME% <%COMPANY_EMAIL%>',
                    array(
                        'NAME'  => 'Foo',
                        'EMAIL' => 'foo@example.com',
                        'COMPANY_NAME'  => 'ACME',
                        'COMPANY_EMAIL' => 'acme@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Foo',
                        'acme@example.com' => 'ACME',
                    )
                ),
                'mixed'    => array(
                    '%NAME% <%EMAIL%>, acme@example.com',
                    array(
                        'NAME'  => 'Foo',
                        'EMAIL' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Foo',
                        'acme@example.com'
                    )
                ),
                'accents' => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'NAME'  => 'Fòo',
                        'EMAIL' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Fòo'
                    )
                ),
                'spaces'  => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'NAME'  => 'Foo  Bar',
                        'EMAIL' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'Foo  Bar'
                    )
                ),
                'underscore' => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'NAME'  => 'Foo Bar',
                        'EMAIL' => 'foo_bar@example.com',
                    ),
                    array(
                        'foo_bar@example.com' => 'Foo Bar'
                    )
                ),
                'name=email' => array(
                    '%NAME% <%EMAIL%>',
                    array(
                        'NAME'  => 'foo@example.com',
                        'EMAIL' => 'foo@example.com',
                    ),
                    array(
                        'foo@example.com' => 'foo@example.com'
                    )
                ),
            );
        }

        /**
         * @expectedException \InvalidArgumentException
         */
        public function testParseEmailsException()
        {
            $helper = new MailerHelper($this->getSiteMock(), $this->getLocalizationMock());

            $helper->parseEmails('foo@bar.baz', array('foo', 'bar', 'baz'));
        }

        /**
         * @dataProvider emailAddressesProvider
         * @param $addresses
         * @param $result
         */
        public function testGetEmailAddresses($addresses, $result)
        {
            $helper = new MailerHelper($this->getSiteMock(), $this->getLocalizationMock());
            $parsed = $helper->getEmailAddresses($addresses);

            $this->assertCount(count($result), $parsed);
            $this->assertEquals($result, $parsed);
        }

        /**
         * @return array
         */
        public function emailAddressesProvider()
        {
            return array(
                'email'                 => array(
                    'email@example.com',
                    array(
                        'email@example.com',
                    )
                ),
                'email+name'            => array(
                    'Name <email@example.com>',
                    array(
                        'email@example.com' => 'Name',
                    )
                ),
                'multiple emails'       => array(
                    'email@example.com, email2@example.com, email3@example.com',
                    array(
                        'email@example.com',
                        'email2@example.com',
                        'email3@example.com',
                    )
                ),
                'multiple emails+names' => array(
                    'email@example.com, Name<email2@example.com>, Name3 <email3@example.com>',
                    array(
                        'email@example.com',
                        'email2@example.com' => 'Name',
                        'email3@example.com' => 'Name3',
                    )
                ),
                'with special chars' => array(
                    'email@e_xample.com, My nàme<_email{2}@example.com>, Name_Surname <strange.email+address@example.com>',
                    array(
                        'email@e_xample.com',
                        '_email{2}@example.com' => 'My nàme',
                        'strange.email+address@example.com' => 'Name_Surname',
                    )
                ),
            );
        }

        /**
         * @return Site|\PHPUnit_Framework_MockObject_MockObject
         */
        protected function getSiteMock()
        {
            $mock = $this->getMock('Application\\Core\\Entity\\SiteInterface');

            $mock->expects($this->any())->method('getName')->willReturn('ACME Website');
            $mock->expects($this->any())->method('getDomain')->willReturn('www.example.com');
            $mock->expects($this->any())->method('getThemesPath')->willReturn('/content/themes/');
            $mock->expects($this->any())->method('getOption')->will(
                $this->returnCallback(
                    function ($string) {
                        $options = array(
                            'company_email' => 'company@example.com',
                            'company_name'  => 'ACME',
                        );

                        return isset($options[$string]) ? $options[$string] : null;
                    }
                )
            );

            return $mock;
        }

        /**
         * @return Localization|\PHPUnit_Framework_MockObject_MockObject
         */
        protected function getLocalizationMock()
        {
            $mock = $this->getMock('Pongho\\Core\\LocalizationInterface');

            $mock->expects($this->any())->method('get')->will(
                $this->returnCallback(
                    function ($string) {
                        return $string;
                    }
                )
            );

            return $mock;
        }
    }
}
