/**
 * geo-localization.js
 *
 * @author Daniele De Nobili
 *
 * DA FARE:
 *
 * - Campo di ricerca per trovare un luogo più velocemente, come in questo esempio:
 *   http://gmaps-samples-v3.googlecode.com/svn/trunk/geocoder/getlatlng.html
 *   Su Pongho RE ho attivato una ricerca che funziona egregiamente tramite le select provincia / città / zona.
 *
 * - Gestione di più marcatori, utile ad esempio quando una grossa azienda ha più sedi da indicare nella mappa:
 *   invece di mostrare tante per mappe, mostro solo una mappa con tanti marcatori. Il codice è già predisposto per
 *   questa funzione, ma bisogna testare e completare.
 */

/*jslint nomen: true, plusplus: true, passfail: true, browser: true, devel: true */
/*global $, document, google */

var google, Pongho = Pongho || {};

(function () {
    "use strict";

    /**
     * Namespace.
     */
    Pongho = Pongho || {};
    Pongho.Form = Pongho.Form || {};
    Pongho.Form.Field = Pongho.Form.Field || {};

    var Geo = Pongho.Form.Field.Geo = {

        /**
         * Initializer.
         *
         * Applica il plugin 'ponghoFormFieldGeo' ai campi.
         */
        Initializer: function () {
            var
                selector = '.geo-localization',
                el = $(selector),
                tabsPanelId = el.closest('.tabsPanel').attr('id'),
                form = el.closest('.form-wrapper');

            if (el.is(':visible')) {
                el.ponghoFormFieldGeo({});
            }

            // Supporto ai repeater
            function manageOnRepeaterAddRow() {
                $('.repeater').on('row_add', function (event, row) {
                    // Devo rinominare il nome del campo
                    var el = row.find(selector),
                        fieldName = el.data('field-name');

                    fieldName = fieldName.replace('[new]', '[' + row.data('id') + ']');
                    el.data('field-name', fieldName);

                    // Applico il plugin agli elementi della nuova riga
                    el.ponghoFormFieldGeo({});
                });
            }

            manageOnRepeaterAddRow();

            // Supporto alle form ajax
            function manageOnPanelUpdate() {
                form.on('activate', '#' + tabsPanelId, function () {
                    $(selector).ponghoFormFieldGeo({});
                });
            }

            manageOnPanelUpdate();
        },

        /**
         * Plugin.
         *
         * È il core del plugin 'ponghoFormFieldGeo'.
         */
        Plugin: function (el) {
            this.el = el;
            this.$el = $(this.el);

            // Passo l’istanza del plugin all’elemento jQuery in modo che sia disponibile anche da fuori,
            // ad esempio per la ricerca.
            this.$el.data("pongho-form-field-geo", this);

            // Dichiaro alcune variabili interne.
            this.isMultiple = this.$el.data('multiple') || false;
            this.fieldName = this.$el.data('field-name');
            this.fieldCounter = 0;
            this.addresses = [];
            this.markers = [];

            // Genero la mappa e la popolo con i marcatori.
            this._generateMap();
            this._populateMap();

            // Gestisce il comportamento della mappa ed in particolare dei marcatori.
            this.handle();
        },

        /**
         * SingleMarkerHandler.
         *
         * Gestisce la mappa con un solo marcatore.
         */
        SingleMarkerHandler: function (geo) {
            var marker = geo.markers[0];

            // Al 'click' sulla mappa, il marcatore si sposta.
            google.maps.event.addListener(geo.map, 'click', function (event) {
                marker.setPosition(event.latLng);
            });
        },

        /**
         * MultipleMarkersHandler.
         *
         * DA COMPLETARE E DA TESTARE.
         * Ad esempio manca la possibilità di rimuovere i marcatori dalla mappa.
         *
         * Gestisce la mappa con più marcatori.
         */
        MultipleMarkersHandler: function (geo) {
            // Al 'click' sulla mappa viene aggiunto un marcatore.
            google.maps.event.addListener(geo.map, 'click', function (event) {
                geo.addMarker(event.latLng);
            });
        }

    };

    $.extend(Geo.Plugin.prototype, {

        /**
         * Aggiunge un marcatore alla mappa.
         *
         * @param pos 'google.maps.LatLng'
         */
        addMarker: function (pos) {
            var
                geo = this,
                marker = new google.maps.Marker({
                    draggable: true,
                    position: pos,
                    map: this.map
                }),
                fields = {};

            this.markers.push(marker);

            // Aggiungo i campi nascosti che terranno traccia di latitudine e longitudine.
            $.map(['lat', 'lng'], function (type) {
                fields[type] = $('<input />')
                    .attr({
                        name: geo.fieldName + '[' + geo.fieldCounter + '][' + type + ']',
                        type: 'hidden'
                    })
                    .appendTo(geo.$el)
                    .val(pos[type]());
            });

            geo.fieldCounter++;

            // Sfrutto l’evento 'position_changed' per aggiornare i valori nei campi nascosti
            google.maps.event.addListener(marker, 'position_changed', function () {
                var pos = marker.getPosition();

                fields.lat.val(pos.lat());
                fields.lng.val(pos.lng());
            });
        },

        /**
         * Permette di eseguire ricerche sulla mappa.
         *
         * <code>
         *     Geo.search("luogo da cercare");
         * </code>
         */
        search: function (value, moveMarker) {
            this._search(value, moveMarker);
        },

        /**
         * Gestisce il comportamento della mappa.
         */
        handle: function () {
            if (this.isMultiple) {
                Geo.MultipleMarkersHandler(this);
            } else {
                Geo.SingleMarkerHandler(this);
            }
        },

        /**
         * Ridimensiona la mappa.
         */
        resize: function () {
            google.maps.event.trigger(this.map, 'resize');
            this.map.setCenter(this.centerMap);
        },

        /**
         * Seguono metodi privati.
         */

        // Inizializza gli indirizzi
        _initializeAddresses: function () {
            var geo = this;

            this.$el.find('address')
                .each(function () {
                    geo.addresses.push($(this).data());
                });
        },

        // Calcola il centro della mappa
        _calculateCenter: function () {
            this._initializeAddresses();

            var lat = 0, lng = 0;

            $.map(this.addresses, function (address) {
                lat += address.lat;
                lng += address.lng;
            });

            this.centerMap = new google.maps.LatLng(lat, lng);
        },

        // Genera la mappa
        _generateMap: function () {
            this._calculateCenter();

            this.mapContainer = $('<div class="geo-localization-map loading" />').appendTo(this.$el);

            this.map = new google.maps.Map(this.mapContainer[0], {
                zoom: this.$el.data('geo-zoom') || 12,
                center: this.centerMap
            });

            this.mapContainer.removeClass('loading');
        },

        // Aggiunge i marcatori alla mappa
        _populateMap: function () {
            var geo = this;

            $.map(this.addresses, function (address) {
                geo.addMarker(new google.maps.LatLng(address.lat, address.lng));
            });
        },

        // Inizializza la ricerca
        _bootGeoCoder: function () {
            if (this.geocoder === undefined) {
                this.geocoder = new google.maps.Geocoder();

                var input,
                    selector = this.$el.data('search-input'),
                    geo = this;

                if (selector) {
                    input = $(selector);

                    input.on('change', function () {
                        geo._search(input.val());
                    });
                }
            }
        },

        // Esegue una ricerca sulla mappa
        _search: function (value, moveMarker) {
            var geo = this,
                zoom;

            this._bootGeoCoder();

            this.geocoder.geocode({
                address: value,
                partialmatch: true
            }, function (results, status) {
                if (status === 'OK' && results.length > 0) {
                    zoom = geo.map.getZoom();
                    google.maps.event.addListenerOnce(geo.map, 'bounds_changed', function (event) {
                        geo.map.setZoom(zoom);
                    });

                    geo.map.fitBounds(results[0].geometry.viewport);

                    if (moveMarker) {
                        geo.markers[0].setPosition(results[0].geometry.location);
                    }
                } else {
                    throw "Geocode was not successful for the following reason: " + status;
                }
            });
        }

    });

    /**
     * Plugin 'ponghoFormFieldGeo'.
     */
    $.fn.ponghoFormFieldGeo = function (options) {
        return $(this).each(function () {
            var
                el = $(this),
                geo = el.data('geo');

            if (!geo) {
                geo = new Geo.Plugin(this);
                el.data('geo', geo);
            }

            if (options === 'resize') {
                geo.resize();
            }
        });
    };

    /**
     * Carica le API di Google Maps.
     */
    function loadScript() {
        var script = document.createElement("script");

        script.type = "text/javascript";
        script.src = "http://maps.googleapis.com/maps/api/js?sensor=false&callback=Pongho.Form.Field.Geo.Initializer";

        document.body.appendChild(script);
    }

    $(document).ready(loadScript);

}());
