<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Model\NotificationTemplate;

/**
 * Class NotificationTemplateGenerator
 */
class NotificationTemplateGenerator implements NotificationTemplateGeneratorInterface
{
    /**
     * @var NotificationConfig
     */
    protected $config;

    /**
     * @var LanguageInterface[]
     */
    protected $languages;

    /**
     * @param LanguageInterface[] $languages
     * @param string              $config_class
     */
    public function __construct(array $languages, $config_class = null)
    {
        if ($config_class !== null) {
            if (!class_exists($config_class)) {
                throw new \InvalidArgumentException(
                    sprintf(
                        'The class "%s" for the notification config generator does not exists',
                        $config_class
                    )
                );
            }
            $config = new $config_class();
        } else {
            $config = new NotificationConfig();
        }

        $this->config = $config;
        $this->languages = $languages;
    }

    /**
     * {@inheritdoc}
     */
    public function generate($name, $path)
    {
        $this->config->prepare($name, $path);

        $template = new NotificationTemplate(
            array(
                'notification'       => $name,
                'sender_to_admin'    => $this->config->getSenderToAdmin(),
                'recipient_to_admin' => $this->config->getRecipientToAdmin(),
                'reply_to_admin'     => $this->config->getReplyToAdmin(),
                'sender_to_user'     => $this->config->getSenderToUser(),
                'recipient_to_user'  => $this->config->getRecipientToUser(),
                'reply_to_user'      => $this->config->getReplyToUser(),
                'subject_to_admin'   => $this->config->getSubjectToAdmin(),
                'content_to_admin'   => $this->config->getContentToAdmin(),
            )
        );

        foreach ($this->languages as $language) {
            $template->prepareTranslation(
                $language->getId(),
                array(
                    'subject_to_user'   => $this->config->getSubjectToUser($language->getIso()),
                    'content_to_user'   => $this->config->getContentToUser($language->getIso()),
                )
            );
        }

        return $template;
    }
}
