<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

/**
 * Class NotificationConfig
 */
class NotificationConfig implements NotificationConfigInterface
{
    /**
     * @var string
     */
    protected $path;

    /**
     * @var object
     */
    protected $config;

    /**
     * @var array
     */
    protected $translations;

    /**
     * @var bool
     */
    protected $prepared = false;

    /**
     * @param string $notification_name
     * @param string $path
     */
    public function prepare($notification_name, $path)
    {
        $this->path = rtrim($path, '/') . '/';
        $filename = $this->path . $notification_name . '.json';

        if (!file_exists($filename)) {
            throw new \InvalidArgumentException('The file "' . $filename . '" does not exists');
        }

        $this->config = json_decode(file_get_contents($filename));

        if (json_last_error()) {
            throw new \LogicException(
                sprintf(
                    'JSON decode of the notification "%s" returns an error: %s',
                    $notification_name,
                    json_last_error_msg()
                )
            );
        }

        $this->prepareTranslations();

        $this->prepared = true;
    }

    /**
     * Imposta le traduzioni in un modo più semplice da gestire
     */
    protected function prepareTranslations()
    {
        $this->translations = array();

        if (isset($this->config->translations)) {
            foreach ($this->config->translations as $translation) {
                $this->translations[$translation->language] = $translation->fields;
            }
        }
    }

    /**
     * @return string|null
     */
    public function getSenderToAdmin()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->sender_to_admin) && !empty($this->config->sender_to_admin)) {
            return $this->config->sender_to_admin;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getRecipientToAdmin()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->recipient_to_admin) && !empty($this->config->recipient_to_admin)) {
            return $this->config->recipient_to_admin;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getReplyToAdmin()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->reply_to_admin) && !empty($this->config->reply_to_admin)) {
            return $this->config->reply_to_admin;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getSenderToUser()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->sender_to_user) && !empty($this->config->sender_to_user)) {
            return $this->config->sender_to_user;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getRecipientToUser()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->recipient_to_user) && !empty($this->config->recipient_to_user)) {
            return $this->config->recipient_to_user;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getReplyToUser()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->reply_to_user) && !empty($this->config->reply_to_user)) {
            return $this->config->reply_to_user;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getSubjectToAdmin()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->subject_to_admin) && !empty($this->config->subject_to_admin)) {
            return $this->config->subject_to_admin;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getContentToAdmin()
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->config->content_to_admin) && !empty($this->config->content_to_admin)) {
            $filename = $this->path . $this->config->content_to_admin;

            return $this->loadFileContent($filename);
        }

        return null;
    }

    /**
     * @param string $language_iso
     * @return null|string
     */
    public function getSubjectToUser($language_iso)
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->translations[$language_iso]) && isset($this->translations[$language_iso]->subject_to_user)) {
            if (!empty($this->translations[$language_iso]->subject_to_user)) {
                return $this->translations[$language_iso]->subject_to_user;
            }
        }

        if (isset($this->translations['it']) && isset($this->translations['it']->subject_to_user)) {
            if (!empty($this->translations['it']->subject_to_user)) {
                return $this->translations['it']->subject_to_user;
            }
        }

        return null;
    }

    /**
     * @param string $language_iso
     * @return null|string
     */
    public function getContentToUser($language_iso)
    {
        if (!$this->prepared) {
            throw new \BadMethodCallException('The config must be prepared to get its properties');
        }

        if (isset($this->translations[$language_iso]) && isset($this->translations[$language_iso]->content_to_user)) {
            if (!empty($this->translations[$language_iso]->content_to_user)) {
                $filename = $this->path . $this->translations[$language_iso]->content_to_user;

                if ($content = $this->loadFileContent($filename)) {
                    return $content;
                }
            }
        }

        if (isset($this->translations['it']) && isset($this->translations['it']->content_to_user)) {
            if (!empty($this->translations['it']->content_to_user)) {
                return $this->loadFileContent($this->path . $this->translations['it']->content_to_user);
            }
        }

        return null;
    }

    /**
     * @param $filename
     * @return null|string
     */
    protected function loadFileContent($filename)
    {
        if (file_exists($filename) && is_file($filename)) {
            return file_get_contents($filename);
        }

        return null;
    }
}
