<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\RoleInterface;

/**
 * Modello per i ruoli.
 *
 * @property int    $id
 * @property string $name
 * @property string $description
 */
class Role extends Base implements RoleInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'roles';

    /**
     * Indica l’ID del ruolo speciale “Utente non registrato”.
     */
    const USER_NOT_LOGGED = 1;

    /**
     * Indica l’ID del ruolo speciale “Utente registrato”.
     */
    const USER_LOGGED = 2;

    /**
     * Indica l’ID del ruolo speciale “Amministratore”.
     */
    const ADMIN = 3;

    /**
     * Lista dei ruoli speciali.
     *
     * @static
     * @var array
     */
    public static $SPECIAL_ROLES = array(self::USER_NOT_LOGGED, self::USER_LOGGED, self::ADMIN);

    /**
     * @var \Application\Core\Model\Permit[]
     */
    protected $permits;

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Indica se il ruolo è eliminabile.
     *
     * @return bool
     */
    public function isDeletable()
    {
        return !in_array($this->id, static::$SPECIAL_ROLES);
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if (!$this->isDeletable()) {
            return true;
        }

        if ($this->delPermits() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * Restituisce l’elenco dei permessi.
     *
     * @return \Application\Core\Model\Permit[]
     */
    public function getPermits()
    {
        if ($this->permits === null) {
            $options = array(
                'joins'      => 'LEFT JOIN ' . PermitRole::tableName() . ' AS pr ON pr.permit_id = `from`.id',
                'conditions' => array('pr.role_id = ? AND `from`.is_enabled = ?', $this->id, true)
            );

            $this->permits = array();
            foreach (Permit::all($options) as $permit) {
                $this->permits[] = $permit;
            }
        }

        return $this->permits;
    }

    /**
     * Restituisce l’elenco delle chiavi dei permessi associati al ruolo.
     *
     * @return array
     */
    public function getPermitsKeys()
    {
        return array_map(
            function ($permit) {
                return $permit->key;
            },
            $this->getPermits()
        );
    }

    /**
     * Restituisce l’elenco degli ID dei permessi associati al ruolo.
     *
     * @return array
     */
    public function getPermitsIds()
    {
        return array_map(
            function ($permit) {
                return $permit->id;
            },
            $this->getPermits()
        );
    }

    /**
     * Indica se un permesso è associato al ruolo.
     *
     * @param string $key
     * @return bool
     */
    public function hasPermit($key)
    {
        foreach ($this->getPermits() as $permit) {
            if ($permit->key === $key) {
                return true;
            }
        }

        return false;
    }

    /**
     * Imposta i permessi per il ruolo.
     *
     * La lista dei permessi passati può essere una lista di permessi intesi
     * come modello, o anche una lista di ID dei permessi.
     *
     * @param array $permits
     * @return self
     */
    public function setPermits(array $permits)
    {
        // preferisco lavorare con gli ID dei permessi
        if (reset($permits) instanceof Permit) {
            foreach ($permits as &$permit) {
                $permit = $permit->id;
            }
        }

        $permits = array_unique($permits);

        // preferisco operare eseguendo una transazione
        $role = $this;

        self::transaction(
            function () use ($role, $permits) {
                // rimuovo tutti i permessi
                $role->delPermits();

                foreach ($permits as $permit_id) {
                    PermitRole::create(
                        array(
                            'permit_id' => $permit_id,
                            'role_id'   => $role->id,
                        )
                    );
                }
            }
        );

        $this->permits = null;

        return $this;
    }

    /**
     * @param string|Permit $permit_or_permit_key
     * @return $this
     */
    public function addPermit($permit_or_permit_key)
    {
        if ($permit_or_permit_key instanceof Permit) {
            $permit = $permit_or_permit_key;
        } else {
            $permit = Permit::findOrCreate($permit_or_permit_key);
        }

        $permit_role = PermitRole::find(array($permit->id, $this->id));

        if (!$permit_role) {
            PermitRole::create(
                array(
                    'permit_id' => $permit->id,
                    'role_id'   => $this->id,
                )
            );
        }

        return $this;
    }

    /**
     * Elimina le associazioni con i permessi.
     *
     * @return self
     */
    public function delPermits()
    {
        $permits = PermitRole::all(
            array(
                'conditions' => array('role_id = ?', $this->id)
            )
        );

        foreach ($permits as $permit) {
            $permit->delete();
        }

        $this->permits = null;

        return $this;
    }
}
