<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageSiteInterface;
use Application\Core\Model\LanguageSite;

/**
 * Class LanguageSiteManager
 */
class LanguageSiteManager extends AbstractCoreManager implements LanguageSiteManagerInterface
{
    static $model = 'LanguageSite';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findById($language_site_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $language_site_id);

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findByLanguageAndSite($language_id, $site_id)
    {
        $models = array_filter(
            $this->findAll(),
            function (LanguageSite $model) use ($language_id, $site_id) {
                if ($model->getLanguageId() !== $language_id) {
                    return false;
                }

                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );

        if (!empty($models)) {
            return reset($models);
        }

        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function (LanguageSite $model) use ($site_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * @param LanguageSiteInterface|LanguageSite $language_site
     * {@inheritdoc}
     */
    public function save(LanguageSiteInterface $language_site)
    {
        if (!$language_site->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function create($language, $site, $path, $is_enabled = true)
    {
        if (is_numeric($language)) {
            $language_id = intval($language);
        } else {
            $language_id = $language->getId();
        }

        if (is_numeric($site)) {
            $site_id = intval($site);
        } else {
            $site_id = $site->getId();
        }

        $model = new LanguageSite();

        $model->language_id = $language_id;
        $model->site_id = $site_id;
        $model->path = $path;
        $model->is_enabled = $is_enabled;

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function enable(LanguageSiteInterface $language_site)
    {
        $language_site->enable();
        $this->save($language_site);
    }

    /**
     * {@inheritdoc}
     */
    public function disable(LanguageSiteInterface $language_site)
    {
        $language_site->disable();
        $this->save($language_site);
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if (!$this->getCache()->isEmpty(static::$model)) {
            return;
        }

        // fixme: rimuovere le opzioni di default, deve restituire modelli puliti, ma il problema è che altre classi usano LanguageSite come se fosse Language

        /** @var LanguageSite $language_site */
        foreach (LanguageSite::all(LanguageSite::getDefaultModelOptions()) as $language_site) {
            $this->getCache()->set(static::$model, $language_site->getId(), $language_site);
        }
    }
}
