<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageModuleInterface;
use Application\Core\Model\LanguageModule;

/**
 * Class LanguageModuleManager
 */
class LanguageModuleManager extends AbstractCoreManager implements LanguageModuleManagerInterface
{
    /**
     * @var string
     */
    static $model = 'LanguageModule';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySiteAndModule($site_id, $module_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageModuleInterface $model) use ($site_id, $module_id) {
                if ($model->getModuleId() !== $module_id) {
                    return false;
                }

                if ($model->getLanguageSite()->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByModuleSite($module_site_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageModuleInterface $model) use ($module_site_id) {
// Siccome ModuleSite viene ricavato tramite relazione usando il sito di LanguageSite, ha sempre lo stesso sito
//                if ($module_site->getSiteId() !== $language_site->getSiteId()) {
//                    return false;
//                }

                if ($model->getModuleSite()->getId() !== $module_site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageModuleInterface $model) use ($site_id) {
                if ($model->getLanguageSite()->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByModule($module_id)
    {
        // TODO: Implement findAllByModule() method.
        throw new \BadMethodCallException('Not implemented');
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByLanguage($language_id)
    {
        // TODO: Implement findAllByLanguage() method.
        throw new \BadMethodCallException('Not implemented');
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByLanguageSite($language_site_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageModuleInterface $model) use ($language_site_id) {
                if ($model->getLanguageSiteId() !== $language_site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findByLanguageSiteAndModule($language_site_id, $module_id)
    {
        $this->prepare();

        $id = $this->getIdentifier($language_site_id, $module_id);
        $model = $this->getCache()->get(static::$model, $id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The language module "%s" is undefined', $id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findByLanguageAndModuleSite($language_id, $module_site_id)
    {
        $model = array_filter(
            $this->findAll(),
            function(LanguageModuleInterface $model) use ($language_id, $module_site_id) {
                try {
                    $module_site = $model->getModuleSite();
                } catch (\InvalidArgumentException $e) {
                    return false;
                }

                try {
                    $language_site = $model->getLanguageSite();
                } catch (\InvalidArgumentException $e) {
                    return false;
                }

                if ($language_site->getLanguageId() !== $language_id) {
                    return false;
                }

                if ($module_site->getId() !== $module_site_id) {
                    return false;
                }

// Siccome ModuleSite viene ricavato tramite relazione usando il sito di LanguageSite, ha sempre lo stesso sito
//                if ($module_site->getSiteId() !== $language_site->getSiteId()) {
//                    return false;
//                }

                return true;
            }
        );

        if (empty($model)) {
            throw new \InvalidArgumentException(
                sprintf(
                    'The language module with language "%s" and module site "%s" is undefined',
                    $language_id,
                    $module_site_id
                )
            );
        }

        return reset($model);
    }

    /**
     * {@inheritdoc}
     */
    public function findByLanguageAndSiteAndModule($language_id, $site_id, $module_id)
    {
        // TODO: Implement findByLanguageAndSiteAndModule() method.
        throw new \BadMethodCallException('Not implemented');
    }

    /**
     * {@inheritdoc}
     */
    public function create($language_site, $module, $path, $name, $position, $is_enabled = true)
    {
        if (is_numeric($language_site)) {
            $language_site_id = intval($language_site);
        } else {
            $language_site_id = $language_site->getId();
        }

        if (is_numeric($module)) {
            $module_id = intval($module);
        } else {
            $module_id = $module->getId();
        }

        $model = new LanguageModule();

        $model->language_site_id = $language_site_id;
        $model->module_id = $module_id;
        $model->path = $path;
        $model->name = $name;
        $model->position = $position;
        $model->is_enabled = $is_enabled;

        return $model;
    }

    /**
     * @param LanguageModuleInterface|LanguageModule $language_module
     * {@inheritdoc}
     */
    public function save(LanguageModuleInterface $language_module)
    {
        if (!$language_module->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if (!$this->getCache()->isEmpty(static::$model)) {
            return;
        }

        /** @var LanguageModule $model */
        foreach (LanguageModule::all() as $model) {
            $this->getCache()->set(static::$model, $this->getIdentifierFromModel($model), $model);
        }
    }

    /**
     * @param LanguageModuleInterface $language_module
     * @return string
     */
    protected function getIdentifierFromModel(LanguageModuleInterface $language_module)
    {
        return $this->getIdentifier($language_module->getLanguageSiteId(), $language_module->getModuleId());
    }

    /**
     * @param int $language_site_id
     * @param int $module_id
     * @return string
     */
    protected function getIdentifier($language_site_id, $module_id)
    {
        return $language_site_id . ':' . $module_id;
    }
}
