<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Application;
use Application\Core\Model\Manager\Exception\ApplicationManagerNotFoundException;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;

/**
 * ApplicationManager.
 */
class ApplicationManager implements ApplicationManagerInterface
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @var \Application\Core\Model\Application[]
     */
    protected $apps;

    /**
     * @var array
     */
    protected $apps_sites;

    /**
     * @var bool
     */
    protected $db_error = false;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * {@inheritdoc}
     */
    public function allEnabled($site)
    {
        $this->prepare();
        $this->checkDbIntegrity();

        $site_id = $this->getsiteId($site);

        return array_filter($this->apps, function (Application $application) use ($site_id) {
            return isset($this->apps_sites[$application->getName()][$site_id]);
        });
    }

    /**
     * {@inheritdoc}
     */
    public function allInstalled()
    {
        $this->prepare();

        return $this->apps;
    }

    /**
     * {@inheritdoc}
     */
    public function isInstalled($app_name)
    {
        $this->prepare();

        return isset($this->apps[$app_name]);
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabled($app_name, $site)
    {
        $this->prepare();
        $this->checkDbIntegrity();

        $site_id = $this->getsiteId($site);

        return isset($this->apps_sites[$app_name][$site_id]);
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabledInAnySite($app_name)
    {
        $this->prepare();
        $this->checkDbIntegrity();

        return isset($this->apps_sites[$app_name]) && !empty($this->apps_sites[$app_name]);
    }

    /**
     * {@inheritdoc}
     */
    public function createApplication($app_name)
    {
        $this->prepare();

        if (!isset($this->apps[$app_name])) {
            $this->checkDbIntegrity();

            $load_order = $this->connection->fetchColumn('SELECT MAX(load_order) FROM pongho_apps') + 10;

            $this->connection->insert('pongho_apps', ['app_name' => $app_name, 'load_order' => $load_order]);
            $app_id = $this->connection->lastInsertId('pongho_apps_id_seq');

            $this->apps[$app_name] = new Application($app_id, $app_name);
        }

        return $this->apps[$app_name];
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplication($app_name)
    {
        $this->prepare();

        if (!isset($this->apps[$app_name])) {
            return;
        }

        $this->checkDbIntegrity();

        $app = $this->apps[$app_name];
        $this->connection->delete('pongho_apps_sites', ['app_id' => $app->getId()]);
        $this->connection->delete('pongho_apps', ['id' => $app->getId()]);

        unset($this->apps[$app_name]);
        unset($this->apps_sites[$app_name]);
    }

    /**
     * {@inheritdoc}
     */
    public function createApplicationSite($app_name, $site)
    {
        $this->prepare();
        $this->checkDbIntegrity();

        $site_id = $this->getsiteId($site);

        if (!isset($this->apps[$app_name])) {
            throw new ApplicationManagerNotFoundException($app_name);
        }

        if (isset($this->apps_sites[$app_name][$site_id])) {
            return;
        }

        $app = $this->apps[$app_name];

        $this->connection->insert('pongho_apps_sites', ['app_id' => $app->getId(), 'site_id' => $site_id]);

        $this->apps_sites[$app_name][$site_id] = $site_id;
    }

    /**
     * {@inheritdoc}
     */
    public function deleteApplicationSite($app_name, $site)
    {
        $this->prepare();

        if (!isset($this->apps[$app_name])) {
            return;
        }

        $this->checkDbIntegrity();

        $site_id = $this->getsiteId($site);

        if (isset($this->apps_sites[$app_name][$site_id])) {
            $app = $this->apps[$app_name];

            $this->connection->delete('pongho_apps_sites', ['app_id' => $app->getId(), 'site_id' => $site_id]);

            unset($this->apps_sites[$app_name][$site_id]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function findByAppName($app_name)
    {
        $this->prepare();

        if (isset($this->apps[$app_name])) {
            return $this->apps[$app_name];
        }

        return null;
    }

    private function prepare()
    {
        if ($this->apps !== null) {
            return;
        }

        $this->apps = [];
        $this->apps_sites = [];

        try {
            $this->prepareApps();
        } catch (DBALException $e) {
            // Preparo una situazione di fallback per far funzionare
            // almeno Core e Admin e poter aggiustare il database.
            $this->apps['Core'] = new Application(null, 'Core');
            $this->apps['Admin'] = new Application(null, 'Admin');

            $this->db_error = true;
        }
    }

    /**
     * @param int|SiteInterface $site
     * @return int
     */
    private function getSiteId($site)
    {
        if ($site instanceof SiteInterface) {
            return $site->getId();
        } elseif (is_numeric($site)) {
            return (int) $site;
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Expected an integer or an instance of "SiteInterface". Given "%s" instead.',
                    var_to_string($site)
                )
            );
        }
    }

    private function prepareApps()
    {
        $stm = <<<SQL
SELECT a.id, a.app_name, s.site_id
  FROM pongho_apps AS a
       LEFT JOIN pongho_apps_sites AS s ON s.app_id = a.id
 ORDER BY a.load_order
SQL;

        $sth = $this->connection->prepare($stm);
        $sth->execute();

        while ($row = $sth->fetch(\PDO::FETCH_ASSOC)) {
            if (!isset($this->apps[$row['app_name']])) {
                $this->apps[$row['app_name']] = new Application($row['id'], $row['app_name']);
                $this->apps_sites[$row['app_name']] = [];
            }

            if ($row['site_id']) {
                $this->apps_sites[$row['app_name']][$row['site_id']] = $row['site_id'];
            }
        }
    }

    /**
     * @return bool
     * @throws \RuntimeException If the database is broken.
     */
    private function checkDbIntegrity()
    {
        if ($this->db_error) {
            throw new \RuntimeException(
                'Probably, the database is not updated. Please run the updates.'
            );
        }

        return true;
    }
}
