<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Mailer;

use Application\Core\Entity\SiteInterface;
use Pongho\Core\LocalizationInterface;

/**
 * Class MailerHelper
 */
class MailerHelper
{
    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var LocalizationInterface
     */
    protected $localization;

    /**
     * @param SiteInterface         $site
     * @param LocalizationInterface $localization
     */
    public function __construct(SiteInterface $site, LocalizationInterface $localization)
    {
        $this->site = $site;
        $this->localization = $localization;
    }

    /**
     * Restituisce un array di email per il Mailer::send() in base alla configurazione passata.
     *
     * Esempi:
     * <code>
     *      Se $email è: 'email@example.com'
     *      Risulta in: array('email@example.com')
     * </code>
     * Oppure in formato RFC 2822
     * <code>
     *      Se $email è: 'Name <email@example.com>'
     *      Risulta in: array('email@example.com' => 'Name')
     * </code>
     * Con variabili a sostituzione:
     * <code>
     *      Se $email è: '%NAME% <%EMAIL%>' e $vars è: array('NAME' => 'name', 'EMAIL' => 'email@example.com')
     *      Risulta in: array('email@example.com' => 'Name')
     * </code>
     * Sono possibili configurazioni miste quali:
     * <code>
     *      $email: 'foo@acme.com, %NAME1% <%EMAIL1%>, %NAME2% <%EMAIL2%>, %EMAIL3%'
     * </code>
     *
     * @param string $email
     * @param array  $vars
     * @return array
     */
    public function parseEmails($email, array $vars = array())
    {
        if (!is_hash($vars)) {
            throw new \InvalidArgumentException('The "vars" array MUST be a hash');
        }

        $keys = array_map(
            function ($value) {
                return strtoupper("%{$value}%");
            },
            array_keys($vars)
        );

        $values = array_map(
            function ($value) {
                if (is_array($value)) {
                    return '';
                }

                return htmlspecialchars($value);
            },
            array_values($vars)
        );

        $parsed = str_replace($keys, $values, $email);
        $parsed = $this->parse($parsed, $vars);

        return $this->getEmailAddresses($parsed);
    }

    /**
     * Estrae gli indirizzi email da una stringa che può contenerne vari in diverse varianti, separati da virgola.
     *
     * Esempi:
     * <code>
     *      Se $addresses è: 'foo@bar.com, bar baz <bar@baz.it>,boo<boo@boo.net>'
     *      Risulta in: array('foo@bar.com', 'bar@baz.it' => 'bar baz', 'boo@boo.net' => 'boo')
     * </code>
     *
     * @param string $addresses
     * @return array
     */
    public function getEmailAddresses($addresses)
    {
        $allowed_chars = '\!\#\$\%\&\\\'\*\+\-\/\=\?\^\_\`\{\|\}\~\.';

        preg_match_all('/([^<,]*)(<([[:alnum:]@' . $allowed_chars . ']+)>)?/', $addresses, $matches, PREG_SET_ORDER);

        // vecchio metodo (sbagliato) per ricavare l'email dalla stringa "email@domain.com <Name Surname>"
//        preg_match_all('/([[:alnum:]@\-\.]+)([ ]?<(.*?[^>]*)>)?/', $addresses, $matches, PREG_SET_ORDER);

        $recipients = array();
        foreach ($matches as $address) {
            if (empty($address[0])) {
                continue;
            }

            if (isset($address[3])) {
                $recipients[trim($address[3])] = trim($address[1]);
            } else {
                $recipients[] = trim($address[1]);
            }
        }

        return $recipients;
    }

    /**
     * Dato un template ne sostituisce le variabili per l'inserimento nel corpo della e-mail.
     *
     * Per la traduzione delle chiavi si basa sulla lingua corrente del controller che contiene il modulo di contatto.
     *
     * @access public
     * @param string $string
     * @param array  $vars
     * @return string
     */
    public function parse($string, array $vars = array())
    {
        if (empty($string)) {
            return '';
        }

        // Aggiungo le variabili di default
        $vars = array_merge($this->getDefaultVars(), array_change_key_case($vars, CASE_UPPER));

        $replace_language = function (array $matches) {
            return $this->localization->get($matches[1]);
        };

        $replace_site_config = function (array $matches) {
            return $this->site->getOption($matches[1]);
        };

        $replace_var = function (array $matches) use ($vars) {
            return isset($vars[$matches[1]]) ? $vars[$matches[1]] : $matches[0];
        };

        // Sostituisco le variabili
        $string = str_replace('\\', '\\\\', $string);
        $string = str_replace('\'', '\\\'', $string);

        $string = preg_replace_callback("#{L_(.*?)}#", $replace_language, $string);
        $string = preg_replace_callback("#%L_(.*?)%#", $replace_language, $string);

        $string = str_replace("{C_site_name}", $this->site->getName(), $string);
        $string = str_replace("%C_site_name%", $this->site->getName(), $string);

        $string = preg_replace_callback("#{C_(.*?)}#", $replace_site_config, $string);
        $string = preg_replace_callback("#%C_(.*?)%#", $replace_site_config, $string);

        $string = preg_replace_callback("#{(.*?)}#", $replace_var, $string);
        $string = preg_replace_callback("#%(.*?)%#", $replace_var, $string);

        $string = str_replace('\\\\', '\\', $string);
        $string = str_replace('\\\'', '\'', $string);

        return $string;
    }

    /**
     * Restituisce le variabili di default da usare nelle email.
     *
     * @access protected
     * @return array
     */
    protected function getDefaultVars()
    {
        // Autore del sito
        if ( $this->site->getOption('author_website') )
        {
            $author = '<a href="' . $this->site->getOption('author_website') . '" style="' . $this->site->getOption('email_link_style') . '">' . $this->site->getOption('author_name') . '</a>';
        }
        else
        {
            $author = $this->site->getOption('author_name');
        }

        return array(
            'COMPANY_SITE'			=> absolute_url(''),
            'EMAIL_PRIVACY'			=> sprintf($this->localization->get('email_privacy'), $this->site->getOption('company_name')),
            'EMAIL_COPYRIGHT'		=> sprintf($this->localization->get('email_copyright'), $author),
            'EMAIL_DATE_SEND'		=> sprintf($this->localization->get('email_date_send'), date('d/m/Y')),
        );
    }
}
