<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer;

use Application\Core\Installer\Helper\DatabaseSchemaHelper;
use Application\Core\Installer\Helper\UpdatesLocator;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Schema\Schema;

require_once(__DIR__ . '/../../../src/functions.php');

/**
 * PonghoInstaller
 *
 * @todo Quando sarà disponibile, possiamo ottimizzare la procedura tramite l’inserimento multiplo,
 *       utile in particolare per le tabelle delle città e delle traduzioni.
 *       {@see https://github.com/doctrine/dbal/pull/682}
 */
class PonghoInstaller
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    public function install($domain)
    {
        if ($this->isInstalled()) {
            throw new \Exception('Pongho is already installed!');
        }

        $this->buildSchema();
        $this->createDatabaseFunctions();

        $this->populateSettings();
        $this->populateLanguages();
        $this->populateSites($domain);
        $this->populateModules();
        $this->populateApps();
        $this->populateRoles();
        $this->populatePermissions();
        $this->populateZones();
        $this->populateTranslations();
        $this->populateUsers();

        $this->setForeignKeys();
    }

    /**
     * @return bool
     */
    protected function isInstalled()
    {
        $stm = $this->connection->getDatabasePlatform()->getListTablesSQL();
        $sth = $this->connection->query($stm);

        while ($table = $sth->fetchColumn()) {
            if ($table === 'pongho_settings') {
                return true;
            }
        }

        return false;
    }

    protected function buildSchema()
    {
        (new DatabaseSchemaHelper($this->connection))->update(function (Schema $schema) {

            /*
             * Settings
             */
            $settings_table = $schema->createTable('pongho_settings');
            $settings_table->addColumn('key', 'string', ['length' => 50]);
            $settings_table->addColumn('value', 'text', ['length' => 65535, 'default' => '']);
            $settings_table->setPrimaryKey(['key']);

            /*
             * Sites
             */
            $sites_table = $schema->createTable('pongho_sites');
            $sites_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $sites_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $sites_table->addColumn('domain', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('content_path', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('images_path', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('images_url', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('uploads_path', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('uploads_url', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('themes_path', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('themes_url', 'string', ['length' => 200, 'default' => '']);
            $sites_table->addColumn('default_language_id', 'integer', ['unsigned' => true]);
            $sites_table->addColumn('settings', 'text', ['length' => 65535, 'default' => '']);
            $sites_table->setPrimaryKey(['id']);

            /*
             * Apps
             */
            $apps_table = $schema->createTable('pongho_apps');
            $apps_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $apps_table->addColumn('app_name', 'string', ['length' => 50, 'default' => '']);
            $apps_table->addColumn('load_order', 'integer', ['unsigned' => true, 'default' => 0]);
            $apps_table->setPrimaryKey(['id']);

            /*
             * Apps / Sites
             */
            $apps_sites_table = $schema->createTable('pongho_apps_sites');
            $apps_sites_table->addColumn('app_id', 'integer', ['unsigned' => true]);
            $apps_sites_table->addColumn('site_id', 'integer', ['unsigned' => true]);
            $apps_sites_table->setPrimaryKey(['app_id', 'site_id']);
            $apps_sites_table->addForeignKeyConstraint($apps_table, ['app_id'], ['id']);
            $apps_sites_table->addForeignKeyConstraint($sites_table, ['site_id'], ['id']);

            /*
             * Languages
             */
            $languages_table = $schema->createTable('pongho_languages');
            $languages_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $languages_table->addColumn('iso', 'string', ['length' => 50, 'default' => '']);
            $languages_table->addColumn('culture', 'string', ['length' => 10, 'default' => '']);
            $languages_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $languages_table->addColumn('locale', 'string', ['length' => 50, 'default' => '']);
            $languages_table->addColumn('timezone', 'string', ['length' => 50, 'default' => '']);
            $languages_table->setPrimaryKey(['id']);

            /*
             * Modules
             */
            $modules_table = $schema->createTable('pongho_modules');
            $modules_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $modules_table->addColumn('controller', 'string', ['length' => 100, 'default' => '']);
            $modules_table->setPrimaryKey(['id']);

            /*
             * Modules / Sites
             */
            $modules_sites_table = $schema->createTable('pongho_modules_sites');
            $modules_sites_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $modules_sites_table->addColumn('module_id', 'integer', ['unsigned' => true]);
            $modules_sites_table->addColumn('site_id', 'integer', ['unsigned' => true]);
            $modules_sites_table->addColumn('settings', 'text', ['length' => 65535, 'default' => '']);
            $modules_sites_table->addColumn('is_enabled', 'boolean', ['default' => false]);
            $modules_sites_table->setPrimaryKey(['id']);
            $modules_sites_table->addForeignKeyConstraint($modules_table, ['module_id'], ['id']);
            $modules_sites_table->addForeignKeyConstraint($sites_table, ['site_id'], ['id']);

            /*
             * Languages / Sites
             */
            $languages_sites_table = $schema->createTable('pongho_languages_sites');
            $languages_sites_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $languages_sites_table->addColumn('language_id', 'integer', ['unsigned' => true]);
            $languages_sites_table->addColumn('site_id', 'integer', ['unsigned' => true]);
            $languages_sites_table->addColumn('path', 'string', ['length' => 70, 'default' => '']);
            $languages_sites_table->addColumn('is_enabled', 'boolean', ['default' => false]);
            $languages_sites_table->setPrimaryKey(['id']);
            $languages_sites_table->addForeignKeyConstraint($languages_table, ['language_id'], ['id']);
            $languages_sites_table->addForeignKeyConstraint($sites_table, ['site_id'], ['id']);

            /*
             * Languages / Modules
             */
            $languages_modules_table = $schema->createTable('pongho_languages_modules');
            $languages_modules_table->addColumn('language_site_id', 'integer', ['unsigned' => true]);
            $languages_modules_table->addColumn('module_id', 'integer', ['unsigned' => true]);
            $languages_modules_table->addColumn('path', 'string', ['length' => 100, 'default' => '']);
            $languages_modules_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $languages_modules_table->addColumn('position', 'integer', ['unsigned' => true, 'default' => 0]);
            $languages_modules_table->addColumn('is_enabled', 'boolean', ['default' => false]);
            $languages_modules_table->setPrimaryKey(['language_site_id', 'module_id']);
            $languages_modules_table->addForeignKeyConstraint($languages_sites_table, ['language_site_id'], ['id']);
            $languages_modules_table->addForeignKeyConstraint($modules_table, ['module_id'], ['id']);

            /*
             * Files
             */
            $files_table = $schema->createTable('pongho_files');
            $files_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $files_table->addColumn('site_id', 'integer', ['unsigned' => true]);
            $files_table->addColumn('name', 'string', ['length' => 200, 'default' => '']);
            $files_table->addColumn('size', 'integer', ['unsigned' => true, 'default' => 0]);
            $files_table->addColumn('mimetype', 'string', ['length' => 100, 'default' => '']);
            $files_table->addColumn('file', 'string', ['length' => 200, 'default' => '']);
            $files_table->addColumn('path', 'string', ['length' => 50, 'default' => '']);
            $files_table->setPrimaryKey(['id']);
            $files_table->addForeignKeyConstraint($sites_table, ['site_id'], ['id']);

            /*
             * Link Types
             */
            $link_types_table = $schema->createTable('pongho_link_types');
            $link_types_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $link_types_table->addColumn('language_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $link_types_table->addColumn('link_type', 'string', ['length' => 255, 'default' => '']);
            $link_types_table->addColumn('model_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $link_types_table->addColumn('model_path', 'string', ['length' => 255, 'default' => '']);
            $link_types_table->addColumn('handler', 'string', ['length' => 255, 'default' => '']);
            $link_types_table->setPrimaryKey(['id']);
            $link_types_table->addForeignKeyConstraint($languages_table, ['language_id'], ['id']);

            /*
             * Translations
             */
            $translations_table = $schema->createTable('pongho_translations');
            $translations_table->addColumn('key', 'string', ['length' => 50]);
            $translations_table->addColumn('language_site_id', 'integer', ['unsigned' => true]);
            $translations_table->addColumn('application', 'string', ['length' => 50]);
            $translations_table->addColumn('value', 'text', ['length' => 65535, 'default' => '']);
            $translations_table->setPrimaryKey(['key', 'language_site_id']);
            $translations_table->addForeignKeyConstraint($languages_sites_table, ['language_site_id'], ['id']);

            /*
             * Roles
             */
            $roles_table = $schema->createTable('pongho_roles');
            $roles_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $roles_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $roles_table->addColumn('description', 'string', ['length' => 200, 'default' => '']);
            $roles_table->setPrimaryKey(['id']);

            /*
             * Permissions
             */
            $permits_table = $schema->createTable('pongho_permits');
            $permits_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $permits_table->addColumn('key', 'string', ['length' => 50, 'default' => '']);
            $permits_table->addColumn('is_enabled', 'boolean', ['default' => false]);
            $permits_table->setPrimaryKey(['id']);
            $permits_table->addUniqueIndex(['key']);

            /*
             * Permissions / Roles
             */
            $permits_roles_table = $schema->createTable('pongho_permits_roles');
            $permits_roles_table->addColumn('permit_id', 'integer', ['unsigned' => true]);
            $permits_roles_table->addColumn('role_id', 'integer', ['unsigned' => true]);
            $permits_roles_table->setPrimaryKey(['permit_id', 'role_id']);
            $permits_roles_table->addForeignKeyConstraint($permits_table, ['permit_id'], ['id']);
            $permits_roles_table->addForeignKeyConstraint($roles_table, ['role_id'], ['id']);

            /*
             * Countries
             */
            $countries_table = $schema->createTable('pongho_countries');
            $countries_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $countries_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
            $countries_table->addColumn('postal_name', 'string', ['length' => 20, 'default' => '']);
            $countries_table->addColumn('num', 'string', ['length' => 5, 'default' => '']);
            $countries_table->addColumn('code', 'string', ['length' => 2, 'default' => '']);
            $countries_table->addColumn('code3', 'string', ['length' => 3, 'default' => '']);
            $countries_table->addColumn('address_template', 'text', ['length' => 65535, 'default' => '']);
            $countries_table->setPrimaryKey(['id']);

            /*
             * Regions
             */
            $regions_table = $schema->createTable('pongho_regions');
            $regions_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $regions_table->addColumn('country_id', 'integer', ['unsigned' => true]);
            $regions_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $regions_table->addColumn('name_long', 'string', ['length' => 50, 'default' => '']);
            $regions_table->addColumn('url', 'string', ['length' => 50, 'default' => '']);
            $regions_table->setPrimaryKey(['id']);

            /*
             * Provinces
             */
            $provinces_table = $schema->createTable('pongho_provinces');
            $provinces_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $provinces_table->addColumn('region_id', 'integer', ['unsigned' => true]);
            $provinces_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $provinces_table->addColumn('plate', 'string', ['length' => 10, 'default' => '']);
            $provinces_table->setPrimaryKey(['id']);

            /*
             * Cities
             */
            $cities_table = $schema->createTable('pongho_cities');
            $cities_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $cities_table->addColumn('province_id', 'integer', ['unsigned' => true]);
            $cities_table->addColumn('region_id', 'integer', ['unsigned' => true]);
            $cities_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
            $cities_table->setPrimaryKey(['id']);

            /*
             * Users
             *
             * todo: chiave univoca per la colonna 'username'
             * todo: chiave univoca per la colonna 'email'
             * todo: spostare le colonne 'newsletter' e 'newsletter_activation_key' nell’app Newsletter
             * fixme: la colonna 'address2' deve essere text(65535) oppure string(255)
             */
            $users_table = $schema->createTable('pongho_users');
            $users_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $users_table->addColumn('language_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $users_table->addColumn('role_id', 'integer', ['unsigned' => true]);
            $users_table->addColumn('username', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('email', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('password', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('avatar_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $users_table->addColumn('description', 'text', ['length' => 65535, 'default' => '']);
            $users_table->addColumn('activation_key', 'string', ['length' => 32, 'default' => '']);
            $users_table->addColumn('reset_password_key', 'string', ['length' => 32, 'default' => '']);
            $users_table->addColumn('fullname', 'string', ['length' => 100, 'default' => '']);
            $users_table->addColumn('is_active', 'boolean', ['default' => null, 'notnull' => false]);
            $users_table->addColumn('is_founder', 'boolean', ['default' => false]);
            $users_table->addColumn('is_business', 'boolean', ['default' => false]);
            $users_table->addColumn('created_at', 'datetime');
            $users_table->addColumn('updated_at', 'datetime');
            $users_table->addColumn('last_login_at', 'datetime', ['notnull' => false]);
            $users_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('surname', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('company', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('partita_iva', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('codice_fiscale', 'string', ['length' => 16, 'default' => '']);
            $users_table->addColumn('address', 'string', ['length' => 200, 'default' => '']);
            $users_table->addColumn('city', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('postcode', 'string', ['length' => 20, 'default' => '']);
            $users_table->addColumn('province_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $users_table->addColumn('country_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $users_table->addColumn('telephone', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('fax', 'string', ['length' => 50, 'default' => '']);
            $users_table->addColumn('website', 'string', ['length' => 100, 'default' => '']);
            $users_table->addColumn('googleplus', 'string', ['length' => 255, 'default' => '']);
            $users_table->addColumn('newsletter', 'boolean', ['default' => false]);
            $users_table->addColumn('newsletter_activation_key', 'string', ['length' => 32, 'default' => '']);
            $users_table->addColumn('url', 'string', ['length' => 255, 'default' => '']);
            $users_table->addColumn('old_url', 'string', ['length' => 255, 'default' => '']);
            $users_table->addColumn('address2', 'text', ['length' => 255, 'default' => '']);
            $users_table->setPrimaryKey(['id']);
            // $users_table->addUniqueIndex(['username']);
            // $users_table->addUniqueIndex(['email']);
            $users_table->addIndex(['created_at']);
            $users_table->addForeignKeyConstraint($languages_table, ['language_id'], ['id']);
            $users_table->addForeignKeyConstraint($roles_table, ['role_id'], ['id']);
            $users_table->addForeignKeyConstraint($files_table, ['avatar_id'], ['id']);
            $users_table->addForeignKeyConstraint($provinces_table, ['province_id'], ['id']);
            $users_table->addForeignKeyConstraint($countries_table, ['country_id'], ['id']);

            /*
             * Addresses
             *
             * todo: Valutare la relazione con la tabella delle province
             */
            $addresses_table = $schema->createTable('pongho_addresses');
            $addresses_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $addresses_table->addColumn('user_id', 'integer', ['unsigned' => true]);
            $addresses_table->addColumn('name', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('surname', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('address1', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('address2', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('city', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('province_name', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('postcode', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('country_id', 'integer', ['unsigned' => true]);
            $addresses_table->addColumn('telephone', 'string', ['length' => 100, 'default' => '']);
            $addresses_table->addColumn('province_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $addresses_table->setPrimaryKey(['id']);
            $addresses_table->addForeignKeyConstraint($countries_table, ['country_id'], ['id']);
            $addresses_table->addForeignKeyConstraint($users_table, ['user_id'], ['id']);

            /*
             * Users relations
             */
            $users_relations_table = $schema->createTable('pongho_users_relations');
            $users_relations_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $users_relations_table->addColumn('parent_id', 'integer', ['unsigned' => true]);
            $users_relations_table->addColumn('child_id', 'integer', ['unsigned' => true]);
            $users_relations_table->setPrimaryKey(['id']);
            $users_relations_table->addUniqueIndex(['parent_id', 'child_id']);
            $users_relations_table->addForeignKeyConstraint($users_table, ['parent_id'], ['id']);
            $users_relations_table->addForeignKeyConstraint($users_table, ['child_id'], ['id']);

            /*
             * Users portfolio
             */
            $users_portfolio_table = $schema->createTable('pongho_users_portfolio');
            $users_portfolio_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $users_portfolio_table->addColumn('relation_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $users_portfolio_table->addColumn('ancestor_id', 'integer', ['unsigned' => true]);
            $users_portfolio_table->addColumn('descendant_id', 'integer', ['unsigned' => true]);
            $users_portfolio_table->addColumn('depth', 'smallint', ['unsigned' => true, 'default' => 0]);
            $users_portfolio_table->setPrimaryKey(['id']);
            $users_portfolio_table->addUniqueIndex(['relation_id', 'ancestor_id', 'descendant_id']);
            $users_portfolio_table->addForeignKeyConstraint($users_relations_table, ['relation_id'], ['id']);
            $users_portfolio_table->addForeignKeyConstraint($users_table, ['ancestor_id'], ['id']);
            $users_portfolio_table->addForeignKeyConstraint($users_table, ['descendant_id'], ['id']);

            /*
             * Notifications templates
             */
            $notifications_templates_table = $schema->createTable('pongho_notifications_templates');
            $notifications_templates_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $notifications_templates_table->addColumn('site_id', 'integer', ['unsigned' => true]);
            $notifications_templates_table->addColumn('notification', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('sender_to_admin', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('reply_to_admin', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('recipient_to_admin', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('sender_to_user', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('reply_to_user', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('recipient_to_user', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('subject_to_admin', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_table->addColumn('content_to_admin', 'text', ['default' => '']);
            $notifications_templates_table->setPrimaryKey(['id']);
            $notifications_templates_table->addForeignKeyConstraint($sites_table, ['site_id'], ['id']);
            $notifications_templates_table->addUniqueIndex(['site_id', 'notification']);

            /*
             * Notifications templates translations
             */
            $notifications_templates_translations_table = $schema->createTable('pongho_notifications_templates_translations');
            $notifications_templates_translations_table->addColumn('template_id', 'integer', ['unsigned' => true]);
            $notifications_templates_translations_table->addColumn('language_id', 'integer', ['unsigned' => true]);
            $notifications_templates_translations_table->addColumn('subject_to_user', 'string', ['length' => 255, 'default' => '']);
            $notifications_templates_translations_table->addColumn('content_to_user', 'text', ['default' => '']);
            $notifications_templates_translations_table->setPrimaryKey(['template_id', 'language_id']);
            $notifications_templates_translations_table->addForeignKeyConstraint($notifications_templates_table, ['template_id'], ['id']);
            $notifications_templates_translations_table->addForeignKeyConstraint($languages_table, ['language_id'], ['id']);

            /*
             * Groups
             *
             * todo: Valutare lo spostamento nell’app Newsletter
             */
            $groups_table = $schema->createTable('pongho_groups');
            $groups_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $groups_table->addColumn('name', 'string', ['length' => 50, 'default' => '']);
            $groups_table->addColumn('description', 'string', ['length' => 200, 'default' => '']);
            $groups_table->setPrimaryKey(['id']);

            /*
             * Groups / Users
             *
             * todo: Valutare lo spostamento nell’app Newsletter
             */
            $groups_users_table = $schema->createTable('pongho_groups_users');
            $groups_users_table->addColumn('group_id', 'integer', ['unsigned' => true]);
            $groups_users_table->addColumn('user_id', 'integer', ['unsigned' => true]);
            $groups_users_table->setPrimaryKey(['group_id', 'user_id']);
            $groups_users_table->addForeignKeyConstraint($groups_table, ['group_id'], ['id']);
            $groups_users_table->addForeignKeyConstraint($users_table, ['user_id'], ['id']);

            /*
             * Filters
             */
            $filters_table = $schema->createTable('pongho_filters');
            $filters_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
            $filters_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
            $filters_table->addColumn('namespace', 'string', ['length' => 255, 'default' => '']);
            $filters_table->addColumn('filter', 'text', ['length' => 65535, 'default' => '']);
            $filters_table->addColumn('created_at', 'datetime');
            $filters_table->addColumn('updated_at', 'datetime');
            $filters_table->addColumn('user_id', 'integer', ['unsigned' => true, 'notnull' => false]);
            $filters_table->setPrimaryKey(['id']);
            $filters_table->addForeignKeyConstraint($users_table, ['user_id'], ['id']);

            /*
             * archive last views
             */
            $archive_last_views_table = $schema->createTable('pongho_archive_last_views');
            $archive_last_views_table->addColumn('user_id', 'integer', ['unsigned' => true]);
            $archive_last_views_table->addColumn('entity_type', 'string', ['length' => 255]);
            $archive_last_views_table->addColumn('entity_id', 'integer', ['unsigned' => true]);
            $archive_last_views_table->addColumn('updated_at', 'datetime');
            $archive_last_views_table->setPrimaryKey(['user_id', 'entity_type']);
            $archive_last_views_table->addForeignKeyConstraint($users_table, ['user_id'], ['id'], ['onDelete' => 'CASCADE']);
        });
    }

    protected function createDatabaseFunctions()
    {
        $platform = $this->connection->getDatabasePlatform()->getName();

        switch ($platform) {
            case 'mysql':
                $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_mysql_pongho_like.sql'));

                break;

            case 'postgresql':
                $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_DATE_FORMAT.sql'));
                $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_SUBTIME.sql'));
                $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_fixDates.sql'));
                $this->connection->exec(file_get_contents(__DIR__ . '/Resources/function_pgsql_pongho_like.sql'));

                break;

            default:
                throw new \RuntimeException(sprintf('Database platform "%s" is not supported.', $platform));
        }
    }

    protected function setForeignKeys()
    {
        (new DatabaseSchemaHelper($this->connection))->update(function (Schema $schema) {

            $countries_table = $schema->getTable('pongho_countries');
            $regions_table = $schema->getTable('pongho_regions');
            $provinces_table = $schema->getTable('pongho_provinces');
            $cities_table = $schema->getTable('pongho_cities');

            $regions_table->addForeignKeyConstraint($countries_table, ['country_id'], ['id']);
            $provinces_table->addForeignKeyConstraint($regions_table, ['region_id'], ['id']);
            $cities_table->addForeignKeyConstraint($provinces_table, ['province_id'], ['id']);
            $cities_table->addForeignKeyConstraint($regions_table, ['region_id'], ['id']);

        });
    }

    protected function populateSettings()
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $locator = new UpdatesLocator();
        $updates = array_unique(array_merge($locator->locate('Core'), $locator->locate('Admin')));
        sort($updates);

        $max = max($updates);

        $settings = [
            [
                $key_field_name => 'version',
                'value'         => '2.0.1825-alpha',
            ],
            [
                $key_field_name => 'db_version',
                'value'         => $max,
            ],
            [
                $key_field_name => 'installed_updates',
                'value'         => json_encode($updates),
            ],
        ];

        foreach ($settings as $setting) {
            $this->connection->insert(
                'pongho_settings',
                $setting,
                [
                    $key_field_name => \PDO::PARAM_STR,
                    'value'         => \PDO::PARAM_STR,
                ]
            );
        }
    }

    protected function populateLanguages()
    {
        $languages = [
            [
                'iso'      => 'it',
                'culture'  => 'it_IT',
                'name'     => 'Italiano',
                'locale'   => 'it_IT.UTF-8',
                'timezone' => 'Europe/Rome'
            ],
            [
                'iso'      => 'en',
                'culture'  => 'en_GB',
                'name'     => 'English',
                'locale'   => 'en_GB.UTF-8',
                'timezone' => 'Europe/London'
            ],
            [
                'iso'      => 'de',
                'culture'  => 'de_DE',
                'name'     => 'Deutsch',
                'locale'   => 'de_DE.UTF-8',
                'timezone' => 'Europe/Berlin'
            ],
            [
                'iso'      => 'fr',
                'culture'  => 'fr_FR',
                'name'     => 'Français',
                'locale'   => 'fr_FR.UTF-8',
                'timezone' => 'Europe/Paris'
            ],
        ];

        foreach ($languages as $language) {
            $this->connection->insert('pongho_languages', $language);
        }
    }

    protected function populateSites($domain)
    {
        $this->connection->insert(
            'pongho_sites',
            [
                'name'                => 'Pongho',
                'domain'              => $domain,
                'content_path'        => '/content',
                'images_path'         => '/content/images',
                'images_url'          => $domain . '/content/images',
                'uploads_path'        => '/content/uploads',
                'uploads_url'         => $domain . '/content/uploads',
                'themes_path'         => '/content/themes',
                'themes_url'          => $domain . '/content/themes',
                'default_language_id' => 1,
                'settings'            => serialize([]),
            ],
            [
                'name'                => \PDO::PARAM_STR,
                'domain'              => \PDO::PARAM_STR,
                'content_path'        => \PDO::PARAM_STR,
                'images_path'         => \PDO::PARAM_STR,
                'images_url'          => \PDO::PARAM_STR,
                'uploads_path'        => \PDO::PARAM_STR,
                'uploads_url'         => \PDO::PARAM_STR,
                'themes_path'         => \PDO::PARAM_STR,
                'themes_url'          => \PDO::PARAM_STR,
                'default_language_id' => \PDO::PARAM_INT,
                'settings'            => \PDO::PARAM_STR,
            ]
        );
    }

    protected function populateModules()
    {
        $this->connection->insert(
            'pongho_modules',
            [
                'controller' => 'Application\\Core\\Controller\\UserController',
            ],
            [
                'controller' => \PDO::PARAM_STR,
            ]
        );

        $this->connection->insert(
            'pongho_modules_sites',
            [
                'module_id'  => 1,
                'site_id'    => 1,
                'is_enabled' => true,
            ],
            [
                'module_id'  => \PDO::PARAM_INT,
                'site_id'    => \PDO::PARAM_INT,
                'is_enabled' => \PDO::PARAM_BOOL,
            ]
        );

        $this->connection->insert(
            'pongho_languages_sites',
            [
                'site_id'     => 1,
                'language_id' => 1,
                'path'        => '/',
                'is_enabled'  => true,
            ],
            [
                'site_id'     => \PDO::PARAM_INT,
                'language_id' => \PDO::PARAM_INT,
                'path'        => \PDO::PARAM_STR,
                'is_enabled'  => \PDO::PARAM_BOOL,
            ]
        );

        $this->connection->insert(
            'pongho_languages_modules',
            [
                'language_site_id' => 1,
                'module_id'        => 1,
                'path'             => '/user/',
                'name'             => 'Gestione utenti',
                'position'         => 10,
                'is_enabled'       => true,
            ],
            [
                'language_site_id' => \PDO::PARAM_INT,
                'module_id'        => \PDO::PARAM_INT,
                'path'             => \PDO::PARAM_STR,
                'name'             => \PDO::PARAM_STR,
                'position'         => \PDO::PARAM_INT,
                'is_enabled'       => \PDO::PARAM_BOOL,
            ]
        );

    }

    protected function populateApps()
    {
        $apps = [
            [
                'app_name'   => 'Core',
                'load_order' => 1,
            ],
            [
                'app_name'   => 'Admin',
                'load_order' => 2,
            ]
        ];

        foreach ($apps as $app) {
            $this->connection->insert(
                'pongho_apps',
                $app,
                [
                    'app_name'   => \PDO::PARAM_STR,
                    'load_order' => \PDO::PARAM_INT,
                ]
            );

            $app_id = $this->connection->lastInsertId('pongho_apps_id_seq');

            $this->connection->insert(
                'pongho_apps_sites',
                [
                    'app_id'  => $app_id,
                    'site_id' => 1,
                ],
                [
                    'app_id'  => \PDO::PARAM_INT,
                    'site_id' => \PDO::PARAM_INT
                ]
            );
        }
    }

    protected function populateRoles()
    {
        $roles = [
            [
                'name'        => 'Utente non registrato',
                'description' => 'Ruolo speciale per gestire i permessi degli utenti che non hanno eseguito l’accesso al sito.',
            ],
            [
                'name'        => 'Utente registrato',
                'description' => 'È il ruolo che viene assegnato di default agli utenti registrati.',
            ],
            [
                'name'        => 'Amministratore',
                'description' => 'Ruolo amministrativo per l’accesso all’area privata dove l’utente può gestire il sito.',
            ],
        ];

        foreach ($roles as $role) {
            $this->connection->insert(
                'pongho_roles',
                $role,
                [
                    'name'        => \PDO::PARAM_STR,
                    'description' => \PDO::PARAM_STR,
                ]
            );
        }
    }

    /**
     * @todo: spostare 'users.handle_groups' nell’app Newsletter
     */
    protected function populatePermissions()
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $permissions = [
            [
                'key' => 'admin',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.dashboard.widget_visitsreport',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.dashboard.widget_locationsreport',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.dashboard.widget_keywordsreport',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.dashboard.widget_pageviewsreport',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.dashboard.widget_statsreport',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.options',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.options.translations',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'admin.options.notifications',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.admin',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.add',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.edit',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.delete',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.handle_groups',
                'is_enabled' => true,
                'roles' => [3],
            ],
            [
                'key' => 'users.handle_roles',
                'is_enabled' => true,
                'roles' => [],
            ],
            [
                'key' => 'users.handle_assoc', // portfolio
                'is_enabled' => true,
                'roles' => [],
            ],
            [
                'key' => 'users.is_ancestor', // portfolio
                'is_enabled' => true,
                'roles' => [],
            ],
            [
                'key' => 'users.view_all_users', // portfolio
                'is_enabled' => true,
                'roles' => [],
            ],
        ];

        foreach ($permissions as $data) {
            $this->connection->insert(
                'pongho_permits',
                [
                    $key_field_name => $data['key'],
                    'is_enabled'    => $data['is_enabled'],
                ],
                [
                    'name'        => \PDO::PARAM_STR,
                    'description' => \PDO::PARAM_STR,
                ]
            );

            $permission_id = $this->connection->lastInsertId('pongho_permits_id_seq');

            foreach ($data['roles'] as $role_id) {
                $this->connection->insert(
                    'pongho_permits_roles',
                    [
                        'permit_id' => $permission_id,
                        'role_id'   => $role_id,
                    ],
                    [
                        'permit_id' => \PDO::PARAM_INT,
                        'role_id'   => \PDO::PARAM_INT,
                    ]
                );
            }
        }
    }

    protected function populateZones()
    {
        /*
         * Countries
         */
        $stm = 'INSERT INTO pongho_countries (name, code, code3, num, postal_name, address_template) VALUES (?, ?, ?, ?, ?, ?)';
        $sth = $this->connection->prepare($stm);

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/countries.json'), true) as $country) {
            $sth->bindValue(1, $country['name'], \PDO::PARAM_STR);
            $sth->bindValue(2, $country['code'], \PDO::PARAM_STR);
            $sth->bindValue(3, $country['code3'], \PDO::PARAM_STR);
            $sth->bindValue(4, $country['num'], \PDO::PARAM_STR);
            $sth->bindValue(5, $country['postal_name'], \PDO::PARAM_STR);
            $sth->bindValue(6, $country['address_template'], \PDO::PARAM_STR);

            $sth->execute();
        }

        /*
         * Regions
         */
        $stm = 'INSERT INTO pongho_regions (id, country_id, name, name_long, url) VALUES (?, ?, ?, ?, ?)';
        $sth = $this->connection->prepare($stm);

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/regions.json'), true) as $region) {
            $sth->bindValue(1, $region['id'], \PDO::PARAM_INT);
            $sth->bindValue(2, $region['country_id'], \PDO::PARAM_INT);
            $sth->bindValue(3, $region['name'], \PDO::PARAM_STR);
            $sth->bindValue(4, $region['name_long'], \PDO::PARAM_STR);
            $sth->bindValue(5, $region['url'], \PDO::PARAM_STR);

            $sth->execute();
        }

        /*
         * Provinces
         */
        $stm = 'INSERT INTO pongho_provinces (id, region_id, name, plate) VALUES (?, ?, ?, ?)';
        $sth = $this->connection->prepare($stm);

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/provinces.json'), true) as $province) {
            $sth->bindValue(1, $province['id'], \PDO::PARAM_INT);
            $sth->bindValue(2, $province['region_id'], \PDO::PARAM_INT);
            $sth->bindValue(3, $province['name'], \PDO::PARAM_STR);
            $sth->bindValue(4, $province['plate'], \PDO::PARAM_STR);

            $sth->execute();
        }

        /*
         * Cities
         */
        $stm = 'INSERT INTO pongho_cities (id, province_id, region_id, name) VALUES (?, ?, ?, ?)';
        $sth = $this->connection->prepare($stm);

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/cities.json'), true) as $city) {
            $sth->bindValue(1, $city['id'], \PDO::PARAM_INT);
            $sth->bindValue(2, $city['province_id'], \PDO::PARAM_INT);
            $sth->bindValue(3, $city['region_id'], \PDO::PARAM_INT);
            $sth->bindValue(4, $city['name'], \PDO::PARAM_STR);

            $sth->execute();
        }
    }

    protected function populateTranslations()
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $stm = "INSERT INTO pongho_translations (language_site_id, application, {$key_field_name}, value) VALUES (?, ?, ?, ?)";
        $sth = $this->connection->prepare($stm);

        foreach (json_decode(file_get_contents(__DIR__ . '/Resources/translations.json'), true) as $translation) {
            $sth->bindValue(1, $translation['language_site_id'], \PDO::PARAM_INT);
            $sth->bindValue(2, $translation['application'], \PDO::PARAM_STR);
            $sth->bindValue(3, $translation['key'], \PDO::PARAM_STR);
            $sth->bindValue(4, $translation['value'], \PDO::PARAM_STR);

            $sth->execute();
        }
    }

    protected function populateUsers()
    {
        $date = new \DateTime();

        $users = [
            [
                'language_id' => 1,
                'role_id'     => 1,
                'username'    => 'anonymous',
                'email'       => '',
                'password'    => '',
                'is_active'   => true,
                'is_founder'  => false,
                'fullname'    => '',
                'name'        => '',
                'surname'     => '',
                'company'     => '',
                'url'         => '',
                'created_at'  => $date,
                'updated_at'  => $date,
            ],
            [
                'language_id' => 1,
                'role_id'     => 3,
                'username'    => 'metaline',
                'email'       => 'sviluppo@metaline.it',
                'password'    => '09e5049acebf849ecd8f90e3134842ca',
                'is_active'   => true,
                'is_founder'  => true,
                'fullname'    => 'Daniele De Nobili',
                'name'        => 'Daniele',
                'surname'     => 'De Nobili',
                'company'     => 'Meta Line S.r.l.',
                'url'         => $date->format('U') . random(8, '0123456789'),
                'created_at'  => $date,
                'updated_at'  => $date,
            ],
        ];

        foreach ($users as $user) {
            $this->connection->insert(
                'pongho_users',
                $user,
                [
                    'language_id' => \PDO::PARAM_INT,
                    'role_id'     => \PDO::PARAM_INT,
                    'username'    => \PDO::PARAM_STR,
                    'email'       => \PDO::PARAM_STR,
                    'password'    => \PDO::PARAM_STR,
                    'is_active'   => \PDO::PARAM_BOOL,
                    'is_founder'  => \PDO::PARAM_BOOL,
                    'fullname'    => \PDO::PARAM_STR,
                    'name'        => \PDO::PARAM_STR,
                    'surname'     => \PDO::PARAM_STR,
                    'company'     => \PDO::PARAM_STR,
                    'url'         => \PDO::PARAM_STR,
                    'created_at'  => 'datetime',
                    'updated_at'  => 'datetime',
                ]
            );
        }
    }
}
