<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Entity;

use Pongho\Utilities\DateTime;

/**
 * Interface UserDetailsInterface
 */
interface UserDetailsInterface
{
    /**
     * @return int
     */
    public function getId();

    /**
     * @return null|int
     */
    public function getUserId();

    /**
     * @return null|AccountInterface
     */
    public function getAccount();

    /**
     * Restituisce il nome completo dell’utente.
     * Se il nome risulta vuoto verrà utilizzato lo username.
     * È possibile modificare il comportamento di questa funzione tramite l'evento model.pongho_users.name_function.
     *
     * Esempio:
     * <code>
     *   $account->name = 'Paolino';
     *   $account->surname = 'Paperino';
     *   echo $account->name(); // Paolino Paperino
     *   echo $account->name(true); // Paperino Paolino
     * </code>
     *
     * @param boolean $invert Se invertire l’ordine di nome e cognome.
     * @return string
     */
    public function name($invert = false);

    /**
     * @return string
     */
    public function getName();

    /**
     * @return string
     */
    public function getSurname();

    /**
     * @return string
     */
    public function getEmail();

    /**
     * Elimina l’avatar utente.
     *
     * @return boolean
     */
    public function deleteAvatar();

    /**
     * @return int
     */
    public function getAvatarId();

    /**
     * Restituisce il modello dell’avatar.
     *
     * @return ImageInterface
     */
    public function getAvatar();

    /**
     * @return string
     */
    public function getDescription();

    /**
     * @return string
     */
    public function getCompany();

    /**
     * @return string
     */
    public function getTaxCode();

    /**
     * @return string
     */
    public function getVatNumber();

    /**
     * @return string
     */
    public function getAddress();

    /**
     * @return string
     */
    public function getAddress2();

    /**
     * @return string
     */
    public function getCity();

    /**
     * @return string
     */
    public function getPostCode();

    /**
     * @return ProvinceInterface
     */
    public function getProvince();

    /**
     * @return int
     */
    public function getProvinceId();

    /**
     * @return CountryInterface
     */
    public function getCountry();

    /**
     * @return int
     */
    public function getCountryId();

    /**
     * @return string
     */
    public function getTelephone();

    /**
     * @return string
     */
    public function getMobile();

    /**
     * @return string
     */
    public function getFax();

    /**
     * @return string
     */
    public function getWebsite();

    /**
     * @return DateTime
     */
    public function getCreationDate();

    /**
     * @return DateTime
     */
    public function getLastUpdateDate();

    /**
     * Restituisce i genitori diretti dell’utente.
     *
     * @return \Application\Core\Model\Account[]
     *
     * @api
     */
    public function getParents();

    /**
     * Imposta i genitori diretti dell’utente.
     *
     * @param AccountInterface[] $parents
     * @return $this
     *
     * @api
     */
    public function setParents(array $parents);

    /**
     * Imposta un genitore diretto dell’utente.
     *
     * @param int|AccountInterface $parent_id_or_parent_instance
     * @throws \LogicException
     * @return $this
     *
     * @api
     */
    public function setParent($parent_id_or_parent_instance);

    /**
     * Rimuove un genitore diretto dell’utente.
     *
     * @param int|AccountInterface $parent_id_or_instance
     * @return bool
     *
     * @api
     */
    public function delParent($parent_id_or_instance);

    /**
     * Indica se l’utente ha un genitore diretto.
     *
     * @param int $parent_id L'ID del genitore che si vuole controllare.
     * @return bool
     *
     * @api
     */
    public function hasParent($parent_id);

    /**
     * Indica se l'utente ha un certo antenato
     *
     * @param $ancestor_id
     * @return bool
     */
    public function hasAncestor($ancestor_id);

    /**
     * Restituisce i discendenti.
     *
     * @return AccountInterface[]
     */
    public function getDescendants();

    /**
     * Restituisce il numero di discendenti.
     *
     * @return int
     */
    public function countDescendants();

    /**
     * Indica se l'utente ha un certo discendente
     *
     * @param $descendant_id
     * @return bool
     */
    public function hasDescendant($descendant_id);
}
