<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Template\TemplateHelperBuilder;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class TemplatingServiceProvider
 */
class TemplatingServiceProvider implements ServiceProviderInterface
{
    /**
     * Permette di registrare dei servizi aggiuntivi al contenitore che gli viene passato.
     *
     * Questo metodo dovrebbe solo impostare o estendere dei servizi,
     * ma non dovrebbe mai richiederli.
     *
     * @param Container $container
     */
    public function register(Container $container)
    {
        $container->extend('template_engine', array($this, 'extendTemplateEngineService'));
        $container->share('theme_view', array($this, 'getThemeViewService'));
        $container->set('global_view_vars', array($this, 'getGlobalViewVarsService'));

        $container->share('template_helper_builder', array($this, 'getTemplateHelperBuilderService'));
        $container->share('template_head', array($this, 'getTemplateHeaderService'));
        $container->share('template_footer', array($this, 'getTemplateFooterService'));
    }

    /**
     * @param Container $container
     * @deprecated Template\Engine non viene più usato, resta però per retrocompatibilità
     */
    public function extendTemplateEngineService(Container $container)
    {
        $container->get('template_engine')->assignVars($container->get('global_view_vars'));
    }

    /**
     * Restituisce il tema della vista.
     *
     * @param Container $container
     * @return \Pongho\Template\Theme Il tema della vista.
     *
     * @internal
     */
    public function getThemeViewService(Container $container)
    {
        $class = $container->getParameter('template.theme_class');

        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        /** @var \Pongho\Template\Theme $theme */
        $theme = new $class($site->getThemesPath());

        $theme->assignVars($container->get('global_view_vars'));

        return $theme;
    }

    /**
     * Restituisce l’elenco delle variabili globali da impostare nelle viste.
     *
     * @param Container $container
     * @return array
     *
     * @internal
     */
    public function getGlobalViewVarsService(Container $container)
    {
        return array(
            'head'     => $container->get('template_head'),
            'footer'   => $container->get('template_footer'),
            'html'     => $container->get('template_html'),
            'lang'     => $container->get('language'),
            'language' => $container->get('language')->getLanguage(),
            'user'     => $container->get('current_user'),
            'site'     => $container->get('site'),
        );
    }

    /**
     * @param Container $container
     * @return TemplateHelperBuilder
     *
     * @internal
     */
    public function getTemplateHelperBuilderService(Container $container)
    {
        return new TemplateHelperBuilder(
            $container->get('request'),
            $container->get('site'),
            $container->get('language_site')->getLanguage()
        );
    }

    /**
     * @param Container $container
     * @return \Pongho\Template\HeadHelper
     *
     * @internal
     */
    public function getTemplateHeaderService(Container $container)
    {
        /** @var TemplateHelperBuilder $builder */
        $builder = $container->get('template_helper_builder');

        return $builder->buildHeader();
    }

    /**
     * @param Container $container
     * @return \Pongho\Template\FooterHelper
     *
     * @internal
     */
    public function getTemplateFooterService(Container $container)
    {
        /** @var TemplateHelperBuilder $builder */
        $builder = $container->get('template_helper_builder');

        return $builder->buildFooter();
    }
}
