<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Localization;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\Manager\AccountManager;
use Application\Core\Model\Manager\ApplicationManager;
use Application\Core\Model\Manager\CoreManagerRegistry;
use Application\Core\Model\Manager\CoreManagerRegistryCache;
use Application\Core\Model\Manager\NotificationTemplateManager;
use Application\Core\Model\Manager\ProvinceManager;
use Application\Core\Model\Manager\RegionManager;
use Application\Core\Model\Manager\RoleManager;
use Application\Core\Model\Manager\SettingManager;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class ModelManagerProvider
 */
class ModelManagerProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('application_manager', array($this, 'getApplicationManagerService'));
        $container->share('setting_manager', array($this, 'getSettingManagerService'));
        $container->share('account_manager', array($this, 'getAccountManagerService'));
        $container->share('region_manager', array($this, 'getRegionManagerService'));
        $container->share('province_manager', array($this, 'getProvinceManagerService'));
        $container->share('role_manager', array($this, 'getRoleManagerService'));

        // Manager dei modelli base
        $container->share('core_manager_registry', array($this, 'getCoreManagerRegistryService'));
        $container->set('site_manager', array($this, 'getSiteManagerService'));
        $container->set('language_manager', array($this, 'getLanguageManagerService'));
        $container->set('module_manager', array($this, 'getModuleManagerService'));
        $container->set('module_site_manager', array($this, 'getModuleSiteManagerService'));
        $container->set('language_site_manager', array($this, 'getLanguageSiteManagerService'));
        $container->set('language_module_manager', array($this, 'getLanguageModuleManagerService'));

        $container->share('site', array($this, 'getSiteService'));
        $container->share('main_site', array($this, 'getMainSiteService'));
        $container->share('notification_template_manager', array($this, 'getNotificationTemplateManager'));

        $container->share('language_site', array($this, 'getLanguageSiteService'));
        $container->share('language', array($this, 'getLanguageService'));
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\ApplicationManagerInterface
     *
     * @internal
     */
    public function getApplicationManagerService(Container $container)
    {
        return new ApplicationManager($container->get('connection'));
    }

    /**
     * @return \Application\Core\Model\Manager\SettingManager
     *
     * @internal
     */
    public function getSettingManagerService()
    {
        return new SettingManager();
    }

    /**
     * @return AccountManager
     *
     * @internal
     */
    public function getAccountManagerService()
    {
        return new AccountManager();
    }

    /**
     * @return CoreManagerRegistry
     *
     * @internal
     */
    public function getCoreManagerRegistryService()
    {
        return new CoreManagerRegistry(new CoreManagerRegistryCache());
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\SiteManager
     *
     * @internal
     */
    public function getSiteManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getSiteManager();
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\LanguageManager
     *
     * @internal
     */
    public function getLanguageManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getLanguageManager();
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\ModuleManager
     *
     * @internal
     */
    public function getModuleManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getModuleManager();
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\ModuleSiteManager
     *
     * @internal
     */
    public function getModuleSiteManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getModuleSiteManager();
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\LanguageSiteManager
     *
     * @internal
     */
    public function getLanguageSiteManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getLanguageSiteManager();
    }

    /**
     * @param Container $container
     * @return \Application\Core\Model\Manager\LanguageModuleManager
     *
     * @internal
     */
    public function getLanguageModuleManagerService(Container $container)
    {
        return $container->get('core_manager_registry')->getLanguageModuleManager();
    }

    /**
     * Restituisce la lingua.
     *
     * @param Container $container
     * @return \Application\Core\Model\Language|\Application\Core\Model\LanguageSite
     *
     * @internal
     */
    public function getLanguageSiteService(Container $container)
    {
        if (PHP_SAPI === 'cli') {
            /** @var \Application\Core\Model\Site $site */
            $site = $container->get('site');

            return $site->getDefaultLanguage();
        } else {
            /**
             * La lingua può essere decisa in base a 3 parametri:
             *
             * 1. l’indirizzo della pagina
             * 2. le impostazioni dell'utente corrente
             * 3. la lingua impostata nel browser
             *
             * Al terzo parametro ci si arriva solo se l’utente è anonimo. In questo caso, devo controllare
             * se la lingua del browser è disponibile. In alternativa si utilizza quella di default del sito.
             */

            // primo caso, controllo se la rotta ha impostato una lingua (salto se sono in amministrazione o nelle api)
            $request = $container->get('request');
            if (substr($request->getPathInfo(), 0, 8) !== '/pongho/') {
                $router = $container->get('router');
                $parameters = $router->getParameters();

                if (array_key_exists('site_language', $parameters) && $parameters['site_language'] instanceof LanguageSite) {
                    return $parameters['site_language'];
                }
            }

            /**
             * @var \Application\Core\User                              $user
             * @var \Application\Core\Model\Site                        $site
             * @var \Application\Core\Model\Manager\LanguageSiteManager $language_site_manager
             */

            // secondo e terzo caso, vediamo l’utente
            $user = $container->get('current_user');
            $site = $container->get('site');
            $language_site_manager = $container->get('language_site_manager');

            /** @var \Application\Core\Model\Account $account */
            $account = $user->getAccount();

            $user_language_site = $language_site_manager->findByLanguageAndSite(
                $account->language_id,
                $site->getId()
            );

            if ($user->isLogged() && ($user_language_site !== null)) {
                // secondo caso
                return $user_language_site;
            } else {
                // terzo caso
                $request = $container->get('request');

                $languages = array();
                foreach ($site->getLanguages() as $language_site) {
                    $languages[$language_site->getLanguage()->getIso()] = $language_site;
                }

                $preferred = $request->getPreferredLanguage(array_keys($languages));

                if ($preferred && array_key_exists($preferred, $languages)) {
                    // lingua dal browser
                    return $languages[$preferred];
                } else {
                    // lingua di default del sito
                    return $site->getDefaultLanguage();
                }
            }
        }    }

    /**
     * Restituisce il servizio di traduzione
     *
     * @param Container $container
     * @return Localization
     *
     * @internal
     */
    public function getLanguageService(Container $container)
    {
        return new Localization($container->get('language_site'));
    }

    /**
     * Restituisce il modello del sito corrente.
     *
     * @param Container $container
     * @return \Application\Core\Model\Site
     *
     * @internal
     */
    public function getSiteService(Container $container)
    {
        $site_id = $container->getParameter('site_id');

        if (PHP_SAPI !== 'cli') {
            /** @var \Pongho\Http\Request $request */
            $request = $container->get('request');

            if ((substr($request->getPathInfo(), 0, 8) === '/pongho/' || substr($request->getPathInfo(), 0, 5) === '/api/') && $request->query->has('site')) {
                $site_id = $request->query->get('site');
            }
        }

        /** @var \Application\Core\Model\Manager\SiteManager $site_manager */
        $site_manager = $container->get('site_manager');

        $site = $site_manager->findById($site_id);

        if ($site === null) {
            throw new \RuntimeException('Site not found!');
        }

        return $site;
    }

    /**
     * Restituisce il modello del sito principale.
     *
     * @param Container $container
     * @return \Application\Core\Model\Site
     *
     * @internal
     */
    public function getMainSiteService(Container $container)
    {
        try {
            $site = $container->get('site');
        } catch (\RuntimeException $e) {
            // Se non trovo il sito corrente, converto il messaggio dell’eccezione
            throw new \RuntimeException('Main site not found!');
        }

        if ($site->getId() !== 1) {
            /** @var \Application\Core\Model\Manager\SiteManager $site_manager */
            $site_manager = $container->get('site_manager');

            $site = $site_manager->findById(1);

            if ($site === null) {
                throw new \RuntimeException('Main site not found!');
            }
        }

        return $site;
    }

    /**
     * @param Container $container
     * @return NotificationTemplateManager
     *
     * @internal
     */
    public function getNotificationTemplateManager(Container $container)
    {
        return new NotificationTemplateManager(
            $container->get('site'),
            $container->get('notification_collection')
        );
    }

    /**
     * @return RegionManager
     *
     * @internal
     */
    public function getRegionManagerService()
    {
        return new RegionManager();
    }

    /**
     * @return ProvinceManager
     *
     * @internal
     */
    public function getProvinceManagerService()
    {
        return new ProvinceManager();
    }

    /**
     * @return RoleManager
     *
     * @internal
     */
    public function getRoleManagerService()
    {
        return new RoleManager();
    }
}
