<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Monolog\Formatter\HtmlFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\HipChatHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SwiftMailerHandler;
use Monolog\Logger;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class LoggerServiceProvider
 */
class LoggerServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('logger_message', array($this, 'getLoggerMessageService'));
        $container->share('logger', array($this, 'getLoggerService'));
    }

    /**
     * Restituisce il messaggio email per gli errori critici dei log
     *
     * @param Container $container
     * @return \Swift_Message
     *
     * @internal
     */
    public function getLoggerMessageService(Container $container)
    {
        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        $message = new \Swift_Message();

        $message
            ->addFrom($container->getParameter('logger.mail.from', 'noreply@metaline.it'))
            ->addTo($container->getParameter('logger.mail.to', 'log@metaline.it'))
            ->setSubject($site->getName())
            ->setContentType('text/html');

        return $message;
    }

    /**
     * Restituisce il gestore dei log
     *
     * @param Container $container
     * @return \Monolog\Logger
     *
     * @internal
     */
    public function getLoggerService(Container $container)
    {
        $logger_name = 'pongho';
        $logger = new Logger($logger_name);

        /** @var \Application\Core\Model\Site $site */
        $site = $container->get('site');

        // Salvo le informazioni di log su file
        $logger->pushHandler(
            new RotatingFileHandler(
                $site->getContentPath(
                    $container->getParameter('logger.path', '/log') . '/' . $logger_name . '.log'
                ),
                $container->getParameter('logger.maxfiles', 90),
                Logger::DEBUG
            )
        );

        // Mail per gli errori critici
        $swift_mailer_handler = new SwiftMailerHandler(
            new \Swift_Mailer($container->get('mailer_transport')),
            $container->get('logger_message'),
            Logger::DEBUG
        );

        $swift_mailer_handler->setFormatter(new HtmlFormatter());

        $logger->pushHandler(
            new FingersCrossedHandler(
                new BufferHandler(
                    $swift_mailer_handler
                ), Logger::ERROR
            )
        );

        // HipChat nel caso la mail abbia problemi
        $logger->pushHandler(
            new FingersCrossedHandler(
                new BufferHandler(
                    new HipChatHandler(
                        'u10FRj4yUHj7V7zeFZMkUwnEJL6b93yo7ABCwAus',
                        'LOG',
                        '',
                        false,
                        Logger::ERROR,
                        true,
                        true,
                        'text',
                        'api.hipchat.com',
                        HipChatHandler::API_V2
                    )
                ), Logger::ERROR
            )
        );

        // Aggiungo dominio del sito ed utente nei log
        if (PHP_SAPI === 'cli') {
            $user_id = 'cli';
        } else {
            /** @var \Application\Core\User $user */
            $user = $container->get('current_user');
            $user_id = $user->getAccount()->getId();
        }

        $logger->pushProcessor(function (array $record) use ($site, $user_id) {
            $record['extra']['site'] = $site->getDomain();
            $record['extra']['user'] = $user_id;

            return $record;
        });

        return $logger;
    }
}
