<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core\Controller;

use Application\Core\Controller;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Imager\Imager;

/**
 * Controller per la gestione delle immagini.
 */
class ImagesController extends Controller
{
    /**
     * Gestisce l’azione `display`.
     *
     * Carica la versione originale dell’immagine richiesta, la ridimensiona e la invia al browser.
     * In ambiente di produzione crea anche una copia su fylesystem dell’immagine ridimensionata in modo da non doverla ricreare ogni volta.
     *
     * @access public
     */
    public function displayAction()
    {
        $sizes = $this->getView()->getOption('image_sizes', array());

        if (empty($sizes)) {
            throw new \RuntimeException('Parameter "image_sizes" not exists!');
        }

        $default = array(
            'avatar-settings' => array(100, 100, 'fill')
        );

        $sizes = array_merge($default, $sizes);

        $image_relative_path = $this->getParameter('image_path');

        if (!preg_match(
            '#^(/.*?)/([^/]+)\-(' . implode(
                '|',
                array_map('preg_quote', array_keys($sizes))
            ) . ').(jpg|jpeg|png|gif)$#',
            $image_relative_path,
            $matches
        )
        ) {
            throw new HttpNotFoundException();
        }

        $path = $matches[1];
        $filename = $matches[2] . '.' . $matches[4];
        $size = $sizes[$matches[3]];

        switch (count($size)) {
            case 3:
                list($width, $height, $type) = $size;
                break;

            case 4:
                list($width, $height, $type, $background) = $size;
                break;

            default:
                throw new \RuntimeException(sprintf(
                    'Parameter "image_sizes.%s" in themes is not valid: expected 3 or 4 parameters!',
                    $matches[3]
                ));
        }

        create_path($this->getSite()->getImagesPath($path));

        $source_file = $this->getSite()->getUploadsPath($path . '/' . $filename);
        $dest_file = $this->getSite()->getImagesPath($image_relative_path);

        $imager = Imager::load($source_file);

        if (isset($background)) {
            $background = ltrim($background, '#');
            $imager->background($background);
        }

        $imager->resize($width, $height, $type);

        if ($this->getEnvironment() === 'prod') {
            $imager->save($dest_file);
        }

        return $imager->show();
    }
}
