<?php

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Form\Field\PermitsChecklistViewClosure;
use Application\Core\Model\Permit;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Template\ViewClosure;

class RolesController extends CrudFormController
{
    /**
     * {@inheritdoc}
     */
    function getNamespace()
    {
        return 'roles';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.roles.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return 'Application\\Core\\Model\\Role';
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getContainer()->get('translator')->trans('Roles list');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getContainer()->get('translator')->trans('Add role');
    }

    /**
     * {@inheritdoc}
     */
    public function hasAddButton()
    {
        return $this->getHelper()->userHasPermit('users.handle_roles');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getContainer()->get('translator')->trans('Add the first role');
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        return array(
            array(
                'name'      => 'name',
                'label'     => $translator->trans('Role'),
                'class'     => 'large',
                'orderable' => true,
                'render'    => function ($row) {
                    return '<strong>' . $row->name . '</strong><div class="actions"><span>' . $row->actions . '</span></div>';
                }
            ),
			array(
                'name'      => 'description',
                'label'     => $translator->trans('Description'),
                'class'     => 'main',
            ),
            array(
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ),
        );
    }

    /**
     * {@inheritdoc}
     *
     * @param \Application\Core\Model\Role $row
     */
    public function parseArchiveRow($row)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        $path = $this->getParameter('path');

        $actions = array(
            array($translator->trans('Edit'), 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'),
        );

        if ( $row->isDeletable() )
        {
            $actions[] = array($translator->trans('delete'), 'href' => $this->url("/{$path}/delete/{$row->id}/"), 'class' => 'delete');
        }

        $row->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $parameters = array('subject' => $this->getSubject());

        $options = $this->getDefaultFormsOptions();
        $options = $this->getHelper()->filter($this, $this->getEventNamespace('filter_form_options'), $options, $parameters);

        return $options;
    }

    /**
     * Restituisce le opzioni di base della form.
     *
     * @access protected
     * @return \Application\Admin\Form\FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        $subject = $this->getSubject();

        // Impostazioni principali
        $config = new FormConfig('role', $subject, $this->getHelper()->getLocalization());

        // Struttura
        $config->addBaseStructure($this->getActiveTab(), $this->url(str_replace('/pongho/', '/', $this->getRequest()->getRelativeUri())));

        $config->addField(
            'content/main',
            array(
                'class' => 'Application\\Admin\\Form\\PanelFieldset',
                'label' => $translator->trans('Main'),
            )
        );

        $config->addField(
            'content/main/role',
            array(
                'class'      => 'Pongho\\Form\\Fieldset',
                'attributes' => array('class' => 'panel'),
                'settings'   => array(
                    'hide_label' => true,
                ),
            )
        );

        $config->addFields(
            'content/main/role',
            array(
                'name'        => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'attributes' => array(
                        'class'    => 'input_text',
                        'required' => 'required'
                    ),
                    'label'      => $translator->trans('Name'),
                ),
                'description' => array(
                    'class'      => 'Pongho\\Form\\Field\\TextareaField',
                    'label'      => $translator->trans('Description'),
                ),
            )
        );

        if ($this->action !== 'add') {
            // Permessi associati
            $config->addField(
                'content/permits',
                array(
                    'class' => 'Application\\Admin\\Form\\PanelFieldset',
                    'label' => $translator->trans('Permits'),
                )
            );

            $permits_options = array(
                'conditions' => array('is_enabled = ?', true),
                'order'      => '`key`',
            );

            $lang = $this->getHelper()->getLocalization();

            $permits = array();

            /** @var Permit $permit */
            foreach (Permit::all($permits_options) as $permit) {
                $permit->name = '<strong>' . $lang->get('permit_' . $permit->key) . '</strong>';
                $permits[$permit->id] = $permit;
            }

            $config->addField(
                'content/permits/permits',
                array(
                    'class'    => 'Pongho\\Form\\Field\\ChecklistField',
                    'settings' => array(
                        'permits'      => $permits,
                        'localization' => $lang,
                        'view'         => new ViewClosure(new PermitsChecklistViewClosure($lang, $permits))
                    ),
                )
            );
        }

        return $config;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        if ($model->isNewRecord()) {
            return $translator->trans('Add role');
        }

        return $translator->trans('Edit the role “%title%”', array('%title%' => $model->name));
    }

    /**
     * @return ModelSubject
     */
    protected function getSubject()
    {
        $subject = new ModelSubject($this->getModel());

        $subject
            ->attachGetter('permits', array($this, 'getPermitsCallback'))
            ->attachSetter('permits', array($this, 'setPermitsCallback'));

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * @param ModelSubject $subject
     * @return array
     */
    public function getPermitsCallback(ModelSubject $subject)
    {
        /** @var \Application\Core\Model\Role $model */
        $model = $subject->getModel();

        return $model->getPermitsIds();
    }

    /**
     * @param ModelSubject $subject
     * @param array        $value
     */
    public function setPermitsCallback(ModelSubject $subject, array $value)
    {
        /** @var \Application\Core\Model\Role $model */
        $model = $subject->getModel();

        $model->setPermits($value);
    }

    /**
     * {@inheritdoc}
     */
    public function indexAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::indexAction();
    }

    /**
     * {@inheritdoc}
     */
    public function addAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::addAction();
    }

    /**
     * {@inheritdoc}
     */
    public function editAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::editAction();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::deleteAction();
    }
}
