<?php

use Application\Core\Entity\ImageInterface;
use Pongho\Core\Kernel;

/**
 * Genera l'url relativo.
 *
 * Esempio:
 * <code>
 *   echo url('/user/login/'); // Stampa /user/login/
 *   echo url(''); // Stampa /
 *   echo url('/'); // Stampa /
 *   echo url(); // Stampa /
 * </code>
 *
 * È possibile avere anche il percorso relativo della pagina corrente. Ponendo il caso di essere in /about.html:
 * <code>
 *   echo url(); // Su labs stampa /nomesito/about.html
 * </code>
 *
 * @access public
 * @param string $url
 * @return string
 */
function url($url = null)
{
    static $site_path;

    if ( is_null($site_path) )
    {
        /** @var \Pongho\Http\Request $request */
        $request = Kernel::instance()->getContainer()->get('request');

        $site_path = $request->getBasePath();
    }

    return prepend_url($url, $site_path);
}

/**
 * Genera l'url relativo aggiungendo il percorso della lingua.
 *
 * Ad esempio, se sono nella sezione inglese (percorso '/en/'):
 * <code>
 *   echo lang_url('/user/login/'); // Stampa /en/user/login/
 *   echo lang_url(''); // Stampa /en/
 *   echo lang_url('/'); // Stampa /en/
 *   echo lang_url(); // Errore! Non usare!
 * </code>
 *
 * **A differenza di url() non è possibile trattare con il percorso corrente.**
 *
 * @access public
 * @param string $url
 * @return string
 */
function lang_url($url)
{
    static $lang_path;

    if ( is_null($lang_path) )
    {
        $lang_path = Kernel::instance()->getContainer()->get('language')->getLanguagePath();
    }

    return url($lang_path . ltrim($url, '/'));
}

/**
 * Genera l'url assoluto aggiungendo il percorso della lingua.
 *
 * Ad esempio, se sono nella sezione inglese (percorso '/en/'):
 * <code>
 *   echo absolute_lang_url('/user/login/'); // Stampa http://www.example.com/en/user/login/
 *   echo absolute_lang_url(''); // Stampa http://www.example.com/
 *   echo absolute_lang_url('/'); // Stampa http://www.example.com/en/
 *   echo absolute_lang_url(); // Errore! Non usare!
 * </code>
 *
 * **A differenza di url() non è possibile trattare con il percorso corrente.**
 *
 * @access public
 * @param string $url
 * @return string
 */
function absolute_lang_url($url)
{
    static $lang_path;

    if ( is_null($lang_path) )
    {
        $lang_path = Kernel::instance()->getContainer()->get('language')->getLanguagePath();
    }

    return absolute_url($lang_path . ltrim($url, '/'));
}

/**
 * Genera l'url assoluto.
 *
 * Questa funzione è identica ad url(), con la differenza che aggiunge tutto il percorso del sito comprensivo di dominio.
 *
 * Esempio:
 * <code>
 *   echo absolute_url('/user/login/'); // Stampa http://www.example.com/user/login/
 *   echo absolute_url(''); // Stampa http://www.example.com/
 *   echo absolute_url(); // Stampa http://www.example.com/about.html
 * </code>
 *
 * @access public
 * @param string $url
 * @return string
 */
function absolute_url($url = null)
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return prepend_url($url, $site->domain());
}

/**
 * Genera l'url assoluto per un percorso del tema.
 *
 * @access public
 * @param string $url
 * @return string
 */
function themes_url($url)
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getThemesUrl($url));
}

/**
 * Genera l'url assoluto per una immagine.
 *
 * @access public
 * @param string $url
 * @return string
 */
function images_url($url)
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getImagesUrl($url));
}

/**
 * Genera l'url assoluto per i file nella cartella di Pongho.
 *
 * @access public
 * @param string $url
 * @return string
 */
function pongho_url($url)
{
    $dic = Kernel::instance()->getContainer();

    /** @var \Application\Core\Model\Site $site */
    $site = $dic->get('main_site');

    $path = $site->permalink() . trim($dic->getParameter('pongho_path', 'pongho'), '/');

    return $path . '/' . ltrim($url, '/');
}

/**
 * Genera l'url assoluto per un file caricato.
 *
 * @access public
 * @param string $url
 * @return string
 */
function uploads_url($url)
{
    /** @var \Application\Core\Model\Site $site */
    $site = Kernel::instance()->getContainer()->get('site');

    return absolute_url($site->getUploadsUrl($url));
}

/**
 * Restituisce il percorso di una immagine permettendo di specificare dimensioni e tipo di taglio.
 *
 * @access public
 * @param ImageInterface $image
 * @param string         $image_size
 * @param string         $image_not_found
 * @return string
 */
function src(ImageInterface $image = null, $image_size = 'main', $image_not_found = null)
{
    if ($image === null || !file_exists($image->filepath())) {
        if ($image_not_found === null) {
            return themes_url('/images/not-found-' . $image_size . '.png');
        } else {
            return $image_not_found;
        }
    }

    return $image->getSizePermalink($image_size);
}

/**
 * Aggiunge un percorso all'inizio dell'url.
 *
 * @access private
 * @param string $url
 * @param string $prepend
 * @return string
 */
function prepend_url($url, $prepend)
{
    if ( preg_match('#^((f|ht)tp(s)?\:\/\/|mailto\:|callto\:|\#)#', $url) )
    {
        return $url;
    }

    if ( is_null($url) )
    {
        $url = current_url();
    }

    return $prepend . '/' . ltrim($url, '/');
}

/**
 * Restituisce l’url relativo corrente.
 *
 * @access public
 * @return string
 */
function current_url()
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return $request->getPathInfo();
}

/**
 * Indica se sono in home page.
 *
 * @access public
 * @return boolean
 */
function is_home()
{
    $container = Kernel::instance()->getContainer();

    /**
     * @var \Pongho\Http\Request $request
     * @var \Application\Core\Model\LanguageSite $language
     */
    $request = $container->get('request');
    $language = $container->get('language_site');

    return $request->getPathInfo() === $language->getPath();
}

/**
 * Indica se sono in amministrazione.
 *
 * @access public
 * @return boolean
 */
function is_pongho()
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return (bool) ( substr($request->getPathInfo(), 0, 8) === '/pongho/' );
}

/**
 * Indica se la richiesta è di tipo AJAX.
 *
 * @access public
 * @return boolean
 */
function is_ajax()
{
    /** @var \Pongho\Http\Request $request */
    $request = Kernel::instance()->getContainer()->get('request');

    return $request->isAjax();
}

/**
 * Prepara la password per i dovuti controlli.
 *
 * @access public
 * @param string $password
 * @return string
 */
function clean_password($password)
{
    $password = trim($password);

    if ( empty($password) )
    {
        return '';
    }

    return md5($password);
}

/**
 * @access private
 * @return \Application\Core\I18n\LocaleInterface
 *
 * @internal
 */
function _get_locale()
{
    static $locale;

    if (!$locale) {
        /**
         * @var \Application\Core\Model\LanguageSite  $languageSite
         * @var \Application\Core\I18n\LocaleRegistry $localeRegistry
         */
        $languageSite = Kernel::instance()->getContainer()->get('language_site');
        $localeRegistry = Kernel::instance()->getContainer()->get('i18n.locale_registry');

        $locale = $localeRegistry->getLocale($languageSite->getLanguage()->getCulture());
    }

    return $locale;
}

/**
 * Restituisce un prezzo formattato in base alla localizzazione corrente.
 *
 * @param float $price
 * @param bool  $showCurrency
 * @param bool  $showDecimal
 * @return string
 */
function format_price($price, $showCurrency = true, $showDecimal = true)
{
    return _get_locale()->formatPrice($price, $showCurrency, $showDecimal);
}

/**
 * Converte un prezzo da formattato a float in base alla localizzazione corrente.
 *
 * @param string $price
 * @param string $locale
 * @return float
 */
function parse_price($price, $locale = null)
{
    if ($locale !== null) {
        trigger_error(__FUNCTION__ . '(): Parameter $locale is deprecated', E_USER_DEPRECATED);
    }

    return _get_locale()->parsePrice($price);
}

/**
 * Restituisce il nome del mese in base al numero.
 *
 * <code>
 *   echo get_month(1); // Gennaio
 * </code>
 *
 * @access public
 * @param mixed $n
 * @return mixed|string
 */
function get_month($n)
{
    $n = intval($n);

    $value = "MON_$n";
    if (function_exists('nl_langinfo')) {
        return nl_langinfo(constant($value));
    } else {
        $language = Kernel::instance()->getContainer()->get('language')->getLanguage();

        require_once PONGHO_PATH . "/src/languages/{$language->iso}/functions.php";

        $function_name = "Languages\\{$language->iso}\\nl_langinfo";
        if (function_exists($function_name)) {
            return call_user_func($function_name, $value);
        } else {
            // I sistemi windows non hanno la funzione nl_langinfo, restituisco il numero del mese
            return $value;
        }
    }
}

/**
 * Esegue gli shortcode al testo passato.
 *
 * @access public
 * @param string $text
 * @return string
 */
function do_shortcode($text)
{
    return Kernel::instance()->getContainer()->get('shortcode')->parse($text);
}
