<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Account;
use Application\Core\Model\Country;
use Application\Core\Model\Group;
use Application\Core\Model\Province;
use Application\Core\Model\Role;
use Pongho\Form\Subject\ArrayRowSubject;

/**
 * Class UsersImportHelper
 */
class UsersImportHelper
{
    const FILENAME = 'users-import.csv';

    /**
     * @var array|null
     */
    protected $file_headers;

    /**
     * @var array|null
     */
    protected $preview_data;

    /**
     * @var array|null
     */
    protected $file_data;

    /**
     * @var string
     */
    protected $file_import;
    /**
     * @var string
     */
    protected $file_errors;

    /**
     * @var string
     */
    protected $upload_path;

    /**
     * @var string
     */
    protected $delimiter = ';';

    /**
     * @var string
     */
    protected $enclosure = '"';

    /**
     * @var string
     */
    protected $escape = '\\';

    /** @var \Application\Core\I18n\Translation\Translator */
    protected $translator;

    /**
     * @param string     $path Percorso del file da importare
     * @param Translator $translator
     */
    public function __construct($path, Translator $translator)
    {
        $this->translator = $translator;

        if (!file_exists($path)) {
            create_path($path);
        }

        $this->upload_path = $path;
        $this->file_import = $path . '/' . self::FILENAME;
        $this->file_errors = $path . '/users-import-errors.csv';
    }

    /**
     * @param $delimiter
     */
    public function setDelimiter($delimiter)
    {
        if ($delimiter) {
            $this->delimiter = $delimiter;
        }
    }

    /**
     * @param $enclosure
     */
    public function setEnclosure($enclosure)
    {
        if ($enclosure) {
            $this->enclosure = $enclosure;
        }
    }

    /**
     * @param $escape
     */
    public function setEscape($escape)
    {
        if ($escape) {
            $this->escape = $escape;
        }
    }

    /**
     * @return string
     */
    public function getFilename()
    {
        return self::FILENAME;
    }

    /**
     * @return string
     */
    public function getUploadPath()
    {
        return $this->upload_path;
    }

    /**
     * @throws \Exception
     */
    public function readFile()
    {
        if (!file_exists($this->file_import)) {
            throw new \RuntimeException('The CSV file does not exist');
        }

        $sheet = new \SplFileObject($this->file_import, 'r');
        $sheet->setCsvControl($this->delimiter, $this->enclosure, $this->escape);
        $this->file_headers = $sheet->fgetcsv();

        $this->file_data = array();
        $count = count($this->file_headers);

        while($row = $sheet->fgetcsv()) {
            if (count($row) === $count) {
                $this->file_data[] = array_combine($this->file_headers, $row);
            }
        }
    }

    /**
     * @return array|null
     */
    public function getImportHeaders()
    {
        if ($this->file_headers === null) {
            $this->readFile();
        }

        return $this->file_headers;
    }

    /**
     * @param int $limit
     * @return array
     */
    public function getPreviewData($limit = 10)
    {
        if ($this->file_data === null) {
            $this->readFile();
        }

        $data = array_slice($this->file_data, 0, $limit);

        $subjects = array();
        foreach ($data as $row) {
            $subjects[] = new ArrayRowSubject($row);
        }

        return $subjects;
    }

    /**
     * @return array
     */
    public function getFileData()
    {
        if ($this->file_data === null) {
            $this->readFile();
        }

        return $this->file_data;
    }

    /**
     * @param array $header
     * @return array
     */
    public function createAccounts(array $header)
    {
        $filedata = $this->getFileData();

        // Preparo i gruppi
        $groups = $this->prepareGroups($header);

        $file = $this->file_errors;
        $error_file = new \SplFileObject($file, 'w');
        $error_file->setCsvControl($this->delimiter, $this->enclosure, $this->escape);
        $error_file->fputcsv($this->getImportHeaders());

        $errors = array();
        $count = 0;

        // Ciclo le righe
        foreach ($filedata as $row_number => $row) {
            $attributes = array();
            $user_groups = array();

            // Ciclo le colonne
            foreach ($row as $key => $value) {
                $attribute_key = $header[$key];

                switch ($attribute_key) {
                    case '':
                        // Colonna da ignorare
                        continue;

                    case 'group':
                        // Inserimento dell'utente in un gruppo
                        if ($value) {
                            $user_groups[$key] = true;
                        }
                        break;

                    case 'province':
                        // Ricerca della provincia
                        if (empty($value)) {
                            $attributes['province_id'] = null;
                        } else {
                            $options = array(
                                'conditions' => array(
                                    'name = :province OR plate = :province',
                                    'province' => trim($value)
                                ),
                            );

                            /** @var Province $province */
                            if (($province = Province::first($options)) !== null) {
                                $attributes['province_id'] = $province->id;
                            } else {
                                $error_file->fputcsv($row);
                                $errors[] = array($row_number + 2, $this->translator->trans('Invalid province'), html_escape($value));
                                continue;
                            }
                        }

                        break;

                    case 'country':
                        // Ricerca del Paese
                        if (empty($value)) {
                            $attributes['country_id'] = null;
                        } else {
                            $options = array(
                                'conditions' => array(
                                    'name = :country OR code = :country OR code3 = :country',
                                    'country' => trim($value)
                                ),
                            );

                            /** @var Country $country */
                            if (($country = Country::first($options)) !== null) {
                                $attributes['country_id'] = $country->id;
                            } else {
                                $error = array($row_number + 2, $this->translator->trans('Invalid country'), html_escape($value));
                                $error_file->fputcsv($row);
                                $errors[] = $error;
                                continue;
                            }
                        }

                        break;

                    default:
                        $attributes[$attribute_key] = $value;
                        break;
                }
            }

            if (Account::findByEmail($attributes['email'])) {
                continue;
            }

            // Adattamenti vari
            if (!isset($attributes['username'])) {
                $attributes['username'] = $attributes['email'];
            }

            if (!isset($attributes['password'])) {
                $attributes['password'] = random(8);
            }

            $attributes['privacy'] = true;
            $attributes['newsletter'] = true;
            $attributes['is_active'] = true;
            $attributes['is_founder'] = false;
            $attributes['role_id'] = Role::USER_LOGGED; // todo: multiruolo

            $model = new Account($attributes);

            if ($model->save()) {
                // Gruppi
                foreach ($user_groups as $group_name => $value) {
                    $groups[$group_name]->addUser($model->id);
                }

                $count++;
            } else {
                foreach ($model->getErrors() as $attr => $error) {
                    $error_file->fputcsv($row);
                    $errors[] = array($row_number + 2, $this->translator->get($error), html_escape($attributes[$attr]));
                }
            }
        }

        // Rimuovo il file di importazione
        unlink($this->file_import);

        return array(
            'import_count' => $count,
            'errors'       => $errors,
        );
    }

    /**
     * @param array $header
     * @return \Application\Core\Model\Group[]
     */
    protected function prepareGroups(array $header)
    {
        /** @var \Application\Core\Model\Group[] $groups */
        $groups = array();
        foreach ($header as $column => $attribute_key) {
            if ($attribute_key === 'group') {
                $options = array(
                    'conditions' => array('name = ?', $column),
                );

                if (($group = Group::first($options)) === null) {
                    $group = Group::create(array('name' => $column));
                }

                $groups[$column] = $group;
            }
        }

        return $groups;
    }
}
