<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Utilities;

use Pongho\Core\ControllerHelper as Base;
use Pongho\Core\TemplateResponse;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Template\View;

/**
 * Class ControllerHelper
 */
class ControllerHelper extends Base
{
    /**
     * @var \Application\Core\Model\ModuleSite
     */
    protected $module_site;

    /**
     * Restituisce l’ID del sito corrente.
     *
     * @return integer
     */
    public function getSiteId()
    {
        return $this->container->get('site')->id;
    }

    /**
     * @return string
     */
    public function getPath()
    {
        $path = trim($this->container->getParameter('path'), '/');

        return empty($path) ? '/' : '/' . $path . '/';
    }

    /**
     * @return \Pongho\Template\HeadHelper
     * @deprecated
     */
    public function getHead()
    {
        return $this->getThemeHeaderHelper();
    }

    /**
     * @return \Pongho\Template\HeadHelper
     */
    public function getThemeHeaderHelper()
    {
        return $this->container->get('template_head');
    }

    /**
     * @return \Pongho\Template\FooterHelper
     */
    public function getThemeFooterHelper()
    {
        return $this->container->get('template_footer');
    }

    /**
     * @param string $link
     * @param array  $attributes
     * @return $this
     */
    public function addJavascript($link, array $attributes = [])
    {
        /** @var \Pongho\Template\FooterHelper $templateFooter */
        $templateFooter = $this->container->get('template_footer');

        $templateFooter->javascript->add($link, $attributes);

        return $this;
    }

    /**
     * @param string $code
     * @param array  $attributes
     * @return $this
     */
    public function addJavascriptInline($code, array $attributes = [])
    {
        /** @var \Pongho\Template\FooterHelper $templateFooter */
        $templateFooter = $this->container->get('template_footer');

        $templateFooter->javascript->addInline($code, $attributes);

        return $this;
    }

    /**
     * @param string $link
     * @param array  $attributes
     * @return $this
     */
    public function addCss($link, array $attributes = [])
    {
        /** @var \Pongho\Template\HeadHelper $templateHeader */
        $templateHeader = $this->container->get('template_header');

        $templateHeader->css->add($link, $attributes);

        return $this;
    }

    /**
     * Restituisce la vista.
     *
     * @return \Pongho\Template\Theme
     */
    public function getTheme()
    {
        static $view;

        if ($view === null) {
            $view = $this->filter($this->container->get('controller'), 'core.filter_view', $this->container->get('theme_view'));

            $view->assignVars(
                array(
                    'module' => $this->getModuleSite(false),
                )
            );
        }

        return $view;
    }

    /**
     * Restituisce la vista.
     *
     * @return \Pongho\Template\Theme
     * @deprecated
     */
    public function getView()
    {
        return $this->getTheme();
    }

    /**
     * @param string $template_path
     * @return View
     */
    public function createView($template_path)
    {
        $view = new View($template_path);
        $view->assignVars($this->container->get('global_view_vars'));

        return $view;
    }

    /**
     * Visualizza un messaggio.
     *
     * @param string  $message
     * @param boolean $is_error
     * @param array   $parameters
     * @return TemplateResponse|JsonResponse
     */
    public function displayMessage($message, $is_error = false, array $parameters = array())
    {
        if ($this->container->get('request')->isAjax()) {
            return $this->displayJsonMessage($message, $is_error, $parameters);
        }

        $parameters['message'] = $message;
        $parameters['error'] = $is_error;

        $this->getTheme()
            ->setTemplate('message.php')
            ->assignVars($parameters);

        return new TemplateResponse($this->getTheme(), $this->getResponseHeaders());
    }

    /**
     * Visualizza un messaggio in formato JSON.
     *
     * @param string  $message
     * @param boolean $is_error
     * @param array   $parameters
     * @return JsonResponse
     */
    public function displayJsonMessage($message, $is_error = false, array $parameters = array())
    {
        $parameters['message'] = $message;
        $parameters['error'] = $is_error;

        return new JsonResponse($parameters);
    }

    /**
     * Visualizza un messaggio di errore.
     *
     * @param string $message
     * @param array  $parameters
     * @return JsonResponse
     */
    public function displayError($message, array $parameters = array())
    {
        return $this->displayMessage($message, true, $parameters);
    }

    /**
     * Visualizza un messaggio di errore in formato JSON.
     *
     * @param string $message
     * @param array  $parameters
     * @return JsonResponse
     */
    public function displayJsonError($message, array $parameters = array())
    {
        return $this->displayJsonMessage($message, true, $parameters);
    }

    /**
     * Restituisce una risposta per un redirect all’url passato.
     *
     * Questo metodo tiene conto anche delle richieste AJAX.
     *
     * @param string $redirect_url
     * @return JsonResponse|RedirectResponse
     */
    public function redirectResponse($redirect_url)
    {
        $redirect_url = absolute_url($redirect_url);

        if ($this->container->get('request')->isAjax()) {
            return new JsonResponse(
                array(
                    'redirect' => $redirect_url,
                )
            );
        } else {
            return new RedirectResponse($redirect_url);
        }
    }

    /**
     * Restituisce una risposta di redirect alla pagina di login.
     *
     * Specificando un valore per `$redirect`, dopo il login l’utente verrà reindirizzato alla pagina indicata.
     *
     * @param string $redirect
     * @return JsonResponse|RedirectResponse
     */
    public function redirectToLogin($redirect = null)
    {
        $login_url = rtrim($this->getLanguage()->path, '/') . '/user/login/';

        if ($redirect) {
            $login_url .= '?' . http_build_query(array('redirect' => $redirect));
        }

        return $this->redirectResponse($login_url);
    }

    /**
     * Restituisce la localizzazione.
     *
     * @return \Application\Core\Localization
     */
    public function getLocalization()
    {
        return $this->container->get('language');
    }

    /**
     * Restituisce il modello della lingua.
     *
     * @return \Application\Core\Model\LanguageSite
     */
    public function getLanguage()
    {
        return $this->container->get('language_site');
    }

    /**
     * Restituisce l'id della lingua
     *
     * @return int
     */
    public function getLanguageId()
    {
        return $this->getLanguage()->getLanguageId();
    }

    /**
     * Resituisce il modello del sito corrente.
     *
     * @return \Application\Core\Model\Site
     */
    public function getSite()
    {
        return $this->container->get('site');
    }

    /**
     * Restituisce l’utente.
     *
     * @return \Application\Core\User
     */
    public function getUser()
    {
        return $this->container->get('current_user');
    }

    /**
     * Restituisce l’ID dell'utente corrente.
     *
     * @return int
     */
    public function getUserId()
    {
        return $this->getUser()->getAccount()->getId();
    }

    /**
     * Verifica se l'utente ha un permesso specifico
     *
     * @param string $key
     * @return bool
     */
    public function userHasPermit($key)
    {
        return $this->getUser()->hasPermit($key);
    }

    /**
     * Restituisce il modulo legato al sito.
     *
     * @param boolean $thrown_exception
     * @return \Application\Core\Model\ModuleSite
     */
    public function getModuleSite($thrown_exception = true)
    {
        if ($this->module_site === null) {
            $this->module_site = $this->container->getParameter('site_module', null, $thrown_exception);
        }

        return $this->module_site;
    }

    /**
     * @return \Application\Core\I18n\Translation\Translator
     */
    public function getTranslator()
    {
        return $this->container->get('translator');
    }
}
