<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\Entity\UserDetailsInterface;
use Application\Core\I18n\Translation\Translator;
use Pongho\Utilities\DateTime;

/**
 * Class AccountInfoRenderer
 */
class AccountInfoRenderer
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var UserDetailsInterface
     */
    protected $user_details;

    /**
     * @param Translator $translator
     */
    public function __construct(Translator $translator)
    {
        $this->translator = $translator;
    }

    /**
     * @param UserDetailsInterface $user_details
     */
    public function setAccount(UserDetailsInterface $user_details)
    {
        $this->user_details = $user_details;
    }

    /**
     * @return string
     */
    public function render()
    {
        if ($this->user_details === null) {
            throw new \LogicException('You must set the Account to render its details');
        }

        return '<div id="account-info" class="panel">' . $this->getAvatarCode()
               . '<div id="account-info-content">'
               . $this->getTitleCode()
               . $this->getContentCode()
               . '</div></div>';
    }

    /**
     * @return string
     */
    protected function getAvatarCode()
    {
        $avatar_code = sprintf(
            '<img src="%s">',
            src(
                $this->user_details->getAvatar(),
                'adminpreview',
                pongho_url('/Application/Core/Resources/images/avatar-ghost.png')
            )
        );

        return $avatar_code;
    }

    /**
     * @return string
     */
    protected function getTitleCode()
    {
        $name = $this->user_details->name() ?: $this->translator->trans('New account');

        return '<h3>' . $name . '</h3>';
    }

    /**
     * @return string
     */
    protected function getContentCode()
    {
        return sprintf(
            '<p>%s</p><p>%s</p>',
            $this->translator->trans('Subscription: %date%', array('%date%' => $this->getSubscribeDate())),
            $this->translator->trans('Last login: %date%', array('%date%' => $this->getLastLoginDate()))
        );
    }

    /**
     * @return string
     */
    protected function getLastLoginDate()
    {
        $last_login_date = $this->user_details->getAccount()->getLastLoginDate();

        return ($last_login_date instanceof DateTime)
            ? $last_login_date->format('d-m-Y')
            : $this->translator->trans('never');
    }

    /**
     * @return string
     */
    protected function getSubscribeDate()
    {
        $subscribe_date = $this->user_details->getAccount()->getSubscribeDate();

        if ($subscribe_date instanceof DateTime) {
            return $subscribe_date->format('d-m-Y');
        }

        $date = new DateTime();
        return $date->format('d-m-Y');
    }
}
