<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

// Per il MailerHelper
namespace Application\Core\Mailer
{
    if (!function_exists('Application\\Core\\Mailer\\absolute_url')) {
        /**
         * @param string $url
         * @return string
         */
        function absolute_url($url) {
            return $url;
        }
    }
}

// Per il Mailer
namespace Application\Core\Utilities
{
    if (!function_exists('Application\\Core\\Utilities\\absolute_url')) {
        /**
         * @param string $url
         * @return string
         */
        function absolute_url($url) {
            return $url;
        }
    }
}

namespace Application\Core\Tests\Notification
{
    use Application\Core\Mailer\MailerHelper;
    use Application\Core\Notification\NotificationInterface;
    use Application\Core\Notification\NotificationSender;
    use Application\Core\Notification\UserSubscribeEmailNotification;
    use Application\Core\Tests\Notification\Utilities\MockProvider;

    /**
     * Class UserSubscribeNotificationTest
     */
    class UserSubscribeNotificationTest extends \PHPUnit_Framework_TestCase
    {
        /**
         * @var MockProvider
         */
        protected $mock_provider;

        public function setUp()
        {
            $this->mock_provider = new MockProvider($this);
        }

        public function testNotificationEmailVars()
        {
            $site = $this->mock_provider->getSiteMock();
            $account = $this->mock_provider->getAccountMock();
            $translator = $this->mock_provider->getTranslatorMock();
            $generator = $this->mock_provider->getTemplateGeneratorMock();

            $site->expects($this->any())->method('getOption')->will($this->returnCallback(array($this, 'siteOptionsCallback')));
            $site->expects($this->any())->method('getThemesPath')->will($this->returnCallback(array($this, 'siteThemePathCallback')));
            $site->expects($this->any())->method('getThemesUrl')->will($this->returnCallback(array($this, 'siteThemeUrlCallback')));
            $account->expects($this->once())->method('getUsername')->willReturn('Pippo');
            $account->expects($this->once())->method('getEmail')->willReturn('pippo@email.com');
            $account->expects($this->any())->method('getLanguage')->willReturn($this->mock_provider->getLanguageMock());

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('http://example.com/user/activation/');
            $notification->setSettingsLink('http://example.com/user/settings/');

            $vars = $notification->getEmailVars();

            // Controllo l'esistenza di solo alcuni valori, non mi interessano tutti quelli che va ad inserire il
            // template perché potrebbero essere un'infinità, ma mi interessa che ci siano quelli che imposto nel test
            $expected_vars = array(
                'USER_USERNAME'        => 'Pippo',
                'USER_EMAIL'           => 'pippo@email.com',
                'USER_PASSWORD'        => 'foobar123',
                'USER_ACTIVATION_LINK' => 'http://example.com/user/activation/',
                'USER_SETTINGS_LINK'   => 'http://example.com/user/settings/',
                'PONGHO_COMPANY_LOGO'         => 'http://example.com/theme/images/logo-email.png',
                'PONGHO_COMPANY_NAME'         => 'ACME',
                'PONGHO_COMPANY_EMAIL'        => 'admin@website.com',
            );

            foreach ($expected_vars as $key => $var) {
                $this->assertArrayHasKey($key, $vars);
                $this->assertEquals($var, $vars[$key]);
            }
        }

        public function testSendNotification()
        {
            $site = $this->mock_provider->getSiteMock();
            $account = $this->mock_provider->getAccountMock();
            $mailer = $this->mock_provider->getMailerMock();
            $logger = $this->mock_provider->getLoggerMock();
            $localization = $this->mock_provider->getLocalizationMock();
            $translator = $this->mock_provider->getTranslatorMock();
            $generator = $this->mock_provider->getTemplateGeneratorMock();
            $helper = new MailerHelper($site, $localization);

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('http://example.com/user/activation/');
            $notification->setSettingsLink('http://example.com/user/settings/');

            $template_manager = $this->getTemplateManager($notification);

            // Condizioni
            $site->expects($this->any())->method('getOption')->will($this->returnCallback(array($this, 'siteOptionsCallback')));
            $site->expects($this->any())->method('getName')->willReturn('ACME');
            $site->expects($this->any())->method('getThemesPath')->will($this->returnCallback(array($this, 'siteThemePathCallback')));
            $site->expects($this->any())->method('getThemesUrl')->will($this->returnCallback(array($this, 'siteThemeUrlCallback')));
            $account->expects($this->once())->method('getUsername')->willReturn('Pippo');
            $account->expects($this->once())->method('getEmail')->willReturn('pippo@email.com');
            $account->expects($this->any())->method('getLanguage')->willReturn($this->mock_provider->getLanguageMock());
            $mailer->expects($this->exactly(2))->method('send')->withConsecutive(
                array(
                    array('admin@website.com' => 'ACME'),
                    array('pippo@email.com' => 'Pippo'),
                    $this->equalTo('Welcome!'),
                    $this->equalTo('Your details: Pippo pippo@email.com foobar123')
                ),
                array(
                    array('pippo@email.com' => 'Pippo'),
                    array('admin@website.com' => 'ACME'),
                    $this->equalTo('A new user subscribed!'),
                    $this->equalTo('User details: Pippo pippo@email.com')
                )
            );
            $logger->expects($this->exactly(2))->method('info');

            // Esecuzione
            $sender = new NotificationSender($site, $mailer, $helper, $template_manager, $logger);
            $sender->send($notification);
        }

        /**
         * @expectedException \Application\Core\Notification\Exception\SendException
         * @expectedExceptionMessage Simulated server email send error
         */
        public function testFailSendNotification()
        {
            $site = $this->mock_provider->getSiteMock();
            $account = $this->mock_provider->getAccountMock();
            $mailer = $this->mock_provider->getMailerMock();
            $logger = $this->mock_provider->getLoggerMock();
            $localization = $this->mock_provider->getLocalizationMock();
            $translator = $this->mock_provider->getTranslatorMock();
            $generator = $this->mock_provider->getTemplateGeneratorMock();
            $helper = new MailerHelper($site, $localization);

            $notification = new UserSubscribeEmailNotification($site, $translator, $generator);
            $notification->setAccount($account);
            $notification->setPassword('foobar123');
            $notification->setAccountActivationLink('http://example.com/user/activation/');
            $notification->setSettingsLink('http://example.com/user/settings/');

            $template_manager = $this->getTemplateManager($notification);

            // Condizioni
            $site->expects($this->any())->method('getThemesPath')->will($this->returnCallback(array($this, 'siteThemePathCallback')));
            $site->expects($this->any())->method('getThemesUrl')->will($this->returnCallback(array($this, 'siteThemeUrlCallback')));
            $account->expects($this->once())->method('getUsername');
            $account->expects($this->once())->method('getEmail');
            $account->expects($this->any())->method('getLanguage')->willReturn($this->mock_provider->getLanguageMock());
            $mailer->expects($this->once())->method('send')->willThrowException(new \Exception('Simulated server email send error'));
            $logger->expects($this->once())->method('error');

            // Esecuzione
            $sender = new NotificationSender($site, $mailer, $helper, $template_manager, $logger);
            $sender->send($notification);
        }

        /**
         * @param NotificationInterface $notification
         * @return \Application\Core\Model\Manager\NotificationTemplateManagerInterface|\PHPUnit_Framework_MockObject_MockObject
         */
        protected function getTemplateManager(NotificationInterface $notification)
        {
            return $this->mock_provider->getNotificationTemplateManagerMock(
                array(
                    $notification->getName() => $this->mock_provider->getNotificationTemplateMock(
                        $notification->getName(),
                        '%USER_USERNAME% <%USER_EMAIL%>',
                        '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                        'A new user subscribed!',
                        'User details: %USER_USERNAME% %USER_EMAIL%',
                        '%PONGHO_COMPANY_NAME% <%PONGHO_COMPANY_EMAIL%>',
                        '%USER_USERNAME% <%USER_EMAIL%>',
                        'Welcome!',
                        'Your details: %USER_USERNAME% %USER_EMAIL% %USER_PASSWORD%'
                    ),
                )
            );
        }

        /**
         * @param $option
         * @return mixed
         */
        public function siteOptionsCallback($option)
        {
            $options = array(
                'company_name'  => 'ACME',
                'company_email' => 'admin@website.com',
            );

            if (!isset($options[$option])) {
                return $option;
            }

            return $options[$option];
        }

        /**
         * @param string $file
         * @return string
         */
        public function siteThemePathCallback($file)
        {
            return __DIR__ . '/Fixture/theme/' . $file;
        }

        /**
         * @param string $file
         * @return string
         */
        public function siteThemeUrlCallback($file)
        {
            return 'http://example.com/theme/' . $file;
        }
    }
}
