<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Tests\Breadcrumb;

use Application\Core\Breadcrumb\Breadcrumb;
use Application\Core\Breadcrumb\Crumb;

class BreadcrumbTest extends \PHPUnit_Framework_TestCase
{
    public function testDefault()
    {
        $breadcrumb = $this->buildBreadcrumb();

        $expected = '<li><a href="/foo">foo</a><span aria-hidden="true" class="separator">›</span></li><li class="active">bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    public function testSeparator()
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['separator' => ' / ']);

        $expected = '<li><a href="/foo">foo</a><span aria-hidden="true" class="separator"> / </span></li><li class="active">bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    /**
     * @dataProvider nullProvider
     * @param mixed $separator
     */
    public function testNullSeparator($separator)
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['separator' => $separator]);

        $expected = '<li><a href="/foo">foo</a></li><li class="active">bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    public function testCssClasses()
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions([
                'li_class'        => 'breadcrumb__item',
                'li_active_class' => 'breadcrumb__active',
                'link_class'      => 'breadcrumb__link',
                'name_class'      => 'breadcrumb__name',
                'separator_class' => 'breadcrumb__separator',
            ]);

        $expected = '<li class="breadcrumb__item"><a href="/foo" class="breadcrumb__link"><span class="breadcrumb__name">foo</span></a><span aria-hidden="true" class="breadcrumb__separator">›</span></li><li class="breadcrumb__item breadcrumb__active"><span class="breadcrumb__name">bar</span></li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    /**
     * @dataProvider nullProvider
     * @param mixed $class
     */
    public function testNoSeparatorClass($class)
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['separator_class' => $class]);

        $expected = '<li><a href="/foo">foo</a><span aria-hidden="true">›</span></li><li class="active">bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    /**
     * @dataProvider nullProvider
     * @param mixed $class
     */
    public function testNoLiActiveClass($class)
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['li_active_class' => $class]);

        $expected = '<li><a href="/foo">foo</a><span aria-hidden="true" class="separator">›</span></li><li>bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    public function testSchema()
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['schema' => true]);

        $expected = '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a href="/foo" itemprop="item"><span itemprop="name">foo</span></a><span aria-hidden="true" class="separator">›</span><meta itemprop="position" content="1" /></li><li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem" class="active"><span itemprop="name">bar</span><meta itemprop="position" content="2" /></li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    public function testSchemaPosition()
    {
        $breadcrumb = $this->buildBreadcrumb()
            ->setOptions(['schema' => true, 'schema_position' => 3]);

        $expected = '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a href="/foo" itemprop="item"><span itemprop="name">foo</span></a><span aria-hidden="true" class="separator">›</span><meta itemprop="position" content="3" /></li><li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem" class="active"><span itemprop="name">bar</span><meta itemprop="position" content="4" /></li>';

        $this->assertEquals($expected, $breadcrumb->render());
        $this->assertEquals($expected, (string) $breadcrumb);
    }

    public function testAll()
    {
        $breadcrumb = $this->buildBreadcrumb();

        $this->assertTrue(is_array($breadcrumb->all()));
        $this->assertCount(2, $breadcrumb->all());

        foreach ($breadcrumb->all() as $crumb) {
            $this->assertInstanceOf('Application\Core\Breadcrumb\CrumbInterface', $crumb);
        }
    }

    public function testRemove()
    {
        $breadcrumb = $this->buildBreadcrumb();

        $this->assertCount(2, $breadcrumb->all());

        $breadcrumb->remove(1);
        $all = $breadcrumb->all();

        $this->assertCount(1, $all);
        $crumb = reset($all);

        $this->assertInstanceOf('Application\Core\Breadcrumb\CrumbInterface', $crumb);
        $this->assertEquals(2, $crumb->getId());
    }

    public function testAppendBreadcrumb()
    {
        $breadcrumb = new Breadcrumb();
        $breadcrumb->append(new Crumb(1, 'foo', '/foo'));

        $nested = new Breadcrumb();
        $nested->append(new Crumb(2, 'bar', '/bar'));

        $breadcrumb->append($nested);

        $expected = '<li><a href="/foo">foo</a><span aria-hidden="true" class="separator">›</span></li><li class="active">bar</li>';

        $this->assertEquals($expected, $breadcrumb->render());
    }

    /**
     * @return Breadcrumb
     */
    private function buildBreadcrumb()
    {
        $breadcrumb = new Breadcrumb();

        $breadcrumb->append(new Crumb(1, 'foo', '/foo'));
        $breadcrumb->append(new Crumb(2, 'bar', '/bar'));

        return $breadcrumb;
    }

    public function nullProvider()
    {
        return [
            [null],
            [''],
            [false],
        ];
    }
}
