<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Notification;

use Application\Core\Entity\SiteInterface;
use Application\Core\Mailer\MailerHelper;
use Application\Core\Model\Manager\NotificationTemplateManager;
use Application\Core\Model\Manager\NotificationTemplateManagerInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Notification\Exception\SendException;
use Application\Core\Utilities\Mailer;
use Monolog\Logger;

/**
 * Class NotificationSender
 */
class NotificationSender implements NotificationSenderInterface
{
    /**
     * @var MailerHelper
     */
    protected $helper;

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var NotificationTemplateManager
     */
    protected $template_manager;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @param SiteInterface                        $site
     * @param Mailer                               $mailer
     * @param MailerHelper                         $helper
     * @param NotificationTemplateManagerInterface $template_manager
     * @param Logger                               $logger
     */
    public function __construct(SiteInterface $site, Mailer $mailer, MailerHelper $helper, NotificationTemplateManagerInterface $template_manager, Logger $logger)
    {
        $this->site = $site;
        $this->helper = $helper;
        $this->mailer = $mailer;
        $this->template_manager = $template_manager;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function send(NotificationInterface $notification)
    {
        $vars = $notification->getEmailVars();

        /** @var NotificationTemplate $notification_template */
        $notification_template = $this->template_manager->findByNotification(
            $notification->getName(),
            $notification->getLanguage()
        );

        if ($notification->getOptions()->canSendToUser()) {
            $from = $this->helper->parseEmails($notification_template->getSenderToUser(), $vars);
            $to = $this->helper->parseEmails($notification_template->getRecipientToUser(), $vars);
            $reply_to = $notification->getOptions()->canReplyToUser() ?
                $this->helper->parseEmails($notification_template->getReplyToUser(), $vars) :
                null;

            try {
                $language_id = $notification->getLanguage()->getId();

                if (!$notification_template->hasTranslation($language_id)) {
                    $language_id = $this->site->getDefaultLanguageId();

                    $this->logger->debug(
                        '[Notification] Notification language not found, falling back to site default language',
                        array(
                            'name'      => $notification->getName(),
                            'prev lang' => $notification->getLanguage()->getId(),
                            'new lang'  => $language_id,
                        )
                    );
                }

                if (!$notification_template->hasTranslation($language_id)) {
                    $this->logger->error(
                        '[Notification] Unable to find a suitable fall back language for the notification',
                        array(
                            'name'     => $notification->getName(),
                            'language' => $notification->getLanguage()->getId(),
                        )
                    );

                    throw new \RuntimeException('Unable to find a suitable fall back language for the notification, the web admin has been already notified');
                }

                $this->mailer->send(
                    $from,
                    $to,
                    $this->mailer->parse($notification_template->getSubjectToUser($language_id), $vars),
                    $this->mailer->parse($notification_template->getContentToUser($language_id), $vars),
                    $reply_to
                );

                $this->logger->info(
                    '[Notification] Invio della notifica "' . $notification->getName() . '" all\'utente',
                    array(
                        'from'      => $from,
                        'to'        => $to,
                        'reply_to'  => $reply_to,
                    )
                );
            } catch (\Exception $e) {
                $this->logger->error(
                    '[Notification] Errore invio della notifica "' . $notification->getName() . '" all\'utente',
                    array(
                        'from'      => $from,
                        'to'        => $to,
                        'reply_to'  => $reply_to,
                        'exception' => $e,
                    )
                );

                throw new SendException($e->getMessage(), 'user', $e);
            }
        }

        if ($notification->getOptions()->canSendToAdmin()) {
            $from = $this->helper->parseEmails($notification_template->getSenderToAdmin(), $vars);
            $to = $this->helper->parseEmails($notification_template->getRecipientToAdmin(), $vars);
            $reply_to = $notification->getOptions()->canReplyToAdmin() ?
                $this->helper->parseEmails($notification_template->getReplyToAdmin(), $vars) :
                null;

            try {
                $this->mailer->send(
                    $from,
                    $to,
                    $this->mailer->parse($notification_template->getSubjectToAdmin(), $vars),
                    $this->mailer->parse($notification_template->getContentToAdmin(), $vars),
                    $reply_to
                );

                $this->logger->info(
                    '[Notification] Invio della notifica "' . $notification->getName() . '" all\'amministratore',
                    array(
                        'from'      => $from,
                        'to'        => $to,
                        'reply_to'  => $reply_to,
                    )
                );
            } catch (\Exception $e) {
                $this->logger->error(
                    '[Notification] Errore invio della notifica "' . $notification->getName() . '" all\'amministratore',
                    array(
                        'from'      => $from,
                        'to'        => $to,
                        'reply_to'  => $reply_to,
                        'exception' => $e,
                    )
                );

                throw new SendException($e->getMessage(), 'admin', $e);
            }
        }
    }
}
